package com.onsiteservice.core.config;

import com.alibaba.fastjson.JSON;
import com.onsiteservice.constant.constant.Constants;
import com.onsiteservice.core.exception.CustomExCode;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultCodeEnum;
import com.onsiteservice.core.security.jwt.JwtInterceptor;
import com.onsiteservice.core.util.LogUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.NoHandlerFoundException;
import org.springframework.web.servlet.config.annotation.CorsRegistry;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.PathMatchConfigurer;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import javax.annotation.Resource;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolationException;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;

import static org.springframework.http.HttpStatus.*;

/**
 * @author 潘维吉
 * @date 2018-05-20
 * MVC 核心统一配置
 * 统一异常验证处理、添加拦截器、CORS跨域解决 、请求前缀添加等
 */
@ConditionalOnProperty(prefix = "project.mvc", name = {"enabled"}, matchIfMissing = true)
@Configuration
@Slf4j
public class MvcConfig implements WebMvcConfigurer {

    /** 是否开启统一添加path路径前缀 如/api/** 默认true开启 */
    @Value("${project.path-prefix.enabled:false}")
    private boolean pathPrefixEnabled;
    /** 统一前缀名称 默认/api */
    @Value("${project.path-prefix.name:" + Constants.REQUEST_PREFIX + "}")
    private String pathPrefixName;
    /** 是否开启cors跨域 默认true开启 */
    @Value("${project.cors.enabled:true}")
    private boolean corsEnabled;
    /** cors跨域允许的源 默认*全部 */
    @Value("${project.cors.allowed-origins:*}")
    private String allowedOrigins;

    /** JWT鉴权拦截器 必须注入 否则拦截器里@Value获取值有问题 */
    @Resource
    private JwtInterceptor jwtInterceptor;
    /** 日志工具类处理 */
    @Resource
    private LogUtils logUtils;

    /**
     * 统一异常处理
     */
    @Override
    public void configureHandlerExceptionResolvers(List<HandlerExceptionResolver> exceptionResolvers) {
        exceptionResolvers.add((request, response, handler, exception) -> {
            //统一API响应结果封装类
            Result result = new Result();
            //请求URI
            String requestURI = request.getRequestURI();
            //请求方法
            String requestMethod = request.getMethod();
            //精简异常信息
            String errorMsg = exception.getMessage() == null ? "null空指针异常" : exception.getMessage();
            //是否要统一异常处理
            Boolean isHandle = true;
            //是否打印和存储ServiceException异常日志
            Boolean isLog = true;
            //是否要统一组装异常响应结果信息 用于自定义code和msg情况可跳过
            Boolean isCustomResult = false;

            // 400 业务失败的异常 401 jwt认证失败等，如“账号或密码错误” 或者自定义异常code
            if (exception instanceof ServiceException) {
                ServiceException ex = (ServiceException) exception;
                isHandle = ex.getIsHandle();
                if (!isHandle) {
                    return new ModelAndView();
                }
                isLog = ex.getIsLog();
                // 响应码枚举码
                Integer resultCode = ex.getCode();
                result.setCode(resultCode == null ? ResultCodeEnum.FAIL.getCode() : resultCode);
            }
            // 400 传入参数验证错误异常 或者自定义异常code
            else if (exception instanceof BindException
                    || exception instanceof MethodArgumentNotValidException
                    || exception instanceof ConstraintViolationException
                    || exception instanceof MethodArgumentTypeMismatchException
                    || exception instanceof HttpMessageNotReadableException) {
                result.setCode(ResultCodeEnum.FAIL);

                StringJoiner paramsErrors = new StringJoiner(", ");
                // 遍历所有参数验证错误  统一返回给前端
                List<FieldError> fieldErrorList = new ArrayList<>();
                if (exception instanceof BindException) {
                    fieldErrorList = ((BindException) exception).getBindingResult().getFieldErrors();
                } else if (exception instanceof MethodArgumentNotValidException) {
                    fieldErrorList = ((MethodArgumentNotValidException) exception).getBindingResult().getFieldErrors();
                    // 自定义参数校验错误码和错误信息
                    // 从异常对象中拿到错误信息
                    String defaultMessage = ((MethodArgumentNotValidException) exception).getBindingResult().getAllErrors().get(0).getDefaultMessage();
                    // 参数的Class对象，等下好通过字段名称获取Field对象
                    Class<?> parameterType = ((MethodArgumentNotValidException) exception).getParameter().getParameterType();
                    // 拿到错误的字段名称
                    String fieldName = ((MethodArgumentNotValidException) exception).getBindingResult().getFieldError().getField();
                    Field field = null;
                    try {
                        field = parameterType.getDeclaredField(fieldName);
                    } catch (NoSuchFieldException e) {
                        e.printStackTrace();
                    }
                    // 获取Field对象上的自定义注解
                    CustomExCode annotation = field.getAnnotation(CustomExCode.class);
                    // 有注解的话就返回注解的响应信息
                    if (annotation != null) {
                        result.setCode(annotation.code()).setMsg(annotation.msg()).setData(defaultMessage);
                        isCustomResult = true;
                    }
                } else if (exception instanceof ConstraintViolationException) {
                    paramsErrors.add("参数验证失败: " + exception.getMessage());
                } else if (exception instanceof MethodArgumentTypeMismatchException) {
                    paramsErrors.add("参数类型不匹配: " + exception.getMessage());
                } else {
                    paramsErrors.add("参数异常: " + exception.getMessage());
                }
                for (FieldError fieldError : fieldErrorList) {
                    // 注解验证信息
                    paramsErrors.add(fieldError.getDefaultMessage());
                }
                errorMsg = paramsErrors.toString();
            }
            // 404 找不到资源
            else if (exception instanceof NoHandlerFoundException) {
                result.setCode(ResultCodeEnum.NOT_FOUND);
                errorMsg = requestURI + "接口不存在";
            }
            // 404 Servlet异常 在NoHandlerFoundException之后处理
            else if (exception instanceof ServletException) {
                result.setCode(ResultCodeEnum.NOT_FOUND);
                errorMsg = requestURI + "接口不存在";
            }
            // 500服务器端异常
            else {
                result.setCode(ResultCodeEnum.INTERNAL_SERVER_ERROR);
                errorMsg = requestURI + "接口服务端异常";
            }

            String genMsg = "";
            if (!isCustomResult) { // 自定义响应code和msg跳过
                // 不同环境 不同异常类型 返回不同信息
                genMsg = getExceptionMsg(errorMsg, result);
                result.setMsg(genMsg);
            }

            // response响应捕获的异常结果返回客户端 状态码200
            responseResult(response, result);

            if (isLog && result.getCode() != ResultCodeEnum.NOT_FOUND.getCode()) { // 404 不打印日志
                // 异常信息日志统一组合  打印和存储
                logUtils.exceptionLogs(handler, exception, result, requestMethod, request, genMsg);
            }
            return new ModelAndView();
        });
    }

    /**
     * 统一请求路径前缀添加
     */
    @Override
    public void configurePathMatch(PathMatchConfigurer configurer) {
        if (pathPrefixEnabled) {
            configurer.addPathPrefix(pathPrefixName, c ->
                    c.getPackageName().contains(Constants.PACKAGE_NAME_GROUP) &&
                            (c.isAnnotationPresent(RestController.class) || c.isAnnotationPresent(Controller.class))
            );
        }
    }

    /**
     * 解决CORS(跨域资源共享)跨域
     * 跨域是指浏览器不能执行其他网站的脚本 由浏览器的同源策略造成的 是浏览器对JavaScript实施的安全限制 比如Ajax请求发送不出去  Cookie无法获取等
     * 实现CORS通信的关键是服务器 只要服务器实现了CORS接口就可以跨域通信 浏览器会自动添加一些附加头信息 有时还会多出一次附加请求
     * 非简单请求 如非Get请求 会在正式通信之前增加一次Http请求 预检请求OPTIONS
     */
    @Override
    public void addCorsMappings(CorsRegistry registry) {
        if (corsEnabled) {
            registry.addMapping("/**") //设置允许跨域的路径 **代表所有路径 如 /api/**
                    .allowedOriginPatterns(allowedOrigins) //设置允许跨域请求的域名 *代表所有,可以使用指定的ip,多个的话可以用逗号分隔,默认为*
                    .allowCredentials(true) //是否允许HTTP认证信息 支持证书 允许Cookie 默认为true
                    .allowedMethods("OPTIONS", "GET", "POST", "PUT", "PATCH", "DELETE") //设置允许的请求方法
                    .maxAge(3600) //跨域允许时间 最大过期时间 默认为-1
                    .allowedHeaders("*"); //可访问那些Header信息 JWT认证等使用
        }
    }

    /**
     * 添加注册拦截器
     */
    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        // 多个拦截器时 以此添加 执行顺序按添加顺序
        // addPathPatterns 指定拦截正则匹配模式 统一设置的前缀不在拦截urlPattern里面 填写RequestMapping路径
        // excludePathPatterns 用户排除拦截
        registry.addInterceptor(jwtInterceptor)
                .addPathPatterns("/**");
    }

    /**
     * 响应结果处理
     */
    public static void responseResult(HttpServletResponse response, Result result) {
        response.setCharacterEncoding("UTF-8");
        response.setHeader("Content-type", "application/json;charset=UTF-8");
        //401认证 403授权问题 可修改http返回状态码 默认业务异常都是200
        if (result.getCode() == ResultCodeEnum.UNAUTHORIZED.getCode()) {
            response.setStatus(UNAUTHORIZED.value());
        } else if (result.getCode() == ResultCodeEnum.FORBIDDEN.getCode()) {
            response.setStatus(FORBIDDEN.value());
        } else {
            response.setStatus(OK.value());
        }
        try {
            response.getWriter().write(JSON.toJSONString(result));
        } catch (IOException e) {
            log.error(e.getMessage());
        }
    }

    /**
     * 响应结果处理 http返回状态总是200
     */
    public static void responseHttpOK(HttpServletResponse response, Map map) {
        response.setCharacterEncoding("UTF-8");
        response.setHeader("Content-type", "application/json;charset=UTF-8");
        response.setStatus(OK.value());
        try {
            response.getWriter().write(JSON.toJSONString(map));
        } catch (IOException e) {
            log.error(e.getMessage());
        }
    }

    /**
     * 根据不同异常不同环境生成不同的日志和响应前端的msg信息
     * 生成唯一日志标识 可用于ELK等搜索使用
     */
    private String getExceptionMsg(String errorMsg, Result result) {
        if (result.getCode() == ResultCodeEnum.INTERNAL_SERVER_ERROR.getCode()) {  // 500错误
        /*    SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
            String uuid = sdf.format(new Date()) + (int) ((Math.random() * 9 + 1) * 100000); // 年月日时分秒+随机码(6)
            String msg = "请复制或截屏标识码给系统管理员:" + uuid;*/
            String msg = "服务端出现了错误";
            return msg;
        } else {
            return errorMsg;
        }
    }

}
