package com.onsiteservice.util.aliyun;

import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.model.OSSObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.util.zip.Adler32;
import java.util.zip.CheckedOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * @author 潘维吉
 * @date 2019/10/16 9:27
 * @email 406798106@qq.com
 * @description OSS工具类封装 基于阿里云存储服务
 */
@ConditionalOnProperty(prefix = "aliyun.oss", name = {"enabled"}, matchIfMissing = false)
@Service
@Slf4j
public class OssUtils {

    @Value("${aliyun.access-key-id}")
    private String accessKeyId;
    @Value("${aliyun.access-key-secret}")
    private String accessKeySecret;
    @Value("${aliyun.oss.endpoint:https://oss-cn-beijing.aliyuncs.com}")
    private String endpoint;
    @Value("${aliyun.oss.bucket-name}")
    private String bucketName;
    @Value("${aliyun.oss.cdn-url}")
    private String cdnUrl;

    private OSS ossClient;

    @PostConstruct
    private void initClient() {
        // 创建OSSClient实例
        ossClient = new OSSClientBuilder().build(endpoint, accessKeyId, accessKeySecret);
        // 关闭OSSClient
        //ossClient.shutdown();
    }

    /**
     * 上传文件
     *
     * @param filePath 指定的文件路径 可指定目录(test/1.jpg)
     * @param content  内容 包括流式上传和文件上传。最大不能超过5GB
     */
    public String uploadFile(String filePath, byte[] content) {
        // 上传内容到指定的存储空间（bucketName）并保存为指定的文件名称（filePath）
        ossClient.putObject(bucketName, filePath, new ByteArrayInputStream(content));
        if (cdnUrl.endsWith("/") || filePath.startsWith("/")) {
            return cdnUrl + filePath;
        } else {
            return cdnUrl + "/" + filePath;
        }
    }

    /**
     * 下载文件
     */
    public void downloadFile(String filePath) throws IOException {
        // 调用ossClient.getObject返回一个OSSObject实例，该实例包含文件内容及文件元信息。
        OSSObject ossObject = ossClient.getObject(bucketName, filePath);
        // 调用ossObject.getObjectContent获取文件输入流，可读取此输入流获取其内容。
        InputStream content = ossObject.getObjectContent();
        if (content != null) {
            BufferedReader reader = new BufferedReader(new InputStreamReader(content));
            while (true) {
                String line = reader.readLine();
                if (line == null) {
                    break;
                }
                System.out.println("\n" + line);
            }
            // 数据读取完成后，获取的流必须关闭，否则会造成连接泄漏，导致请求无连接可用，程序无法正常工作。
            content.close();
        }
    }

    /**
     * 删除文件
     */
    public void deleteFile(String filePath) {
        ossClient.deleteObject(bucketName, filePath);
    }

    /**
     * 异步删除图片
     */
    @Async
    public void deleteFileAsync(String filePath) {
        ossClient.deleteObject(bucketName, filePath);
    }

    /**
     * 是否存在存储空间 不存在自动创建存储空间
     */
    private void isExistsBucket() {
        if (!ossClient.doesBucketExist(bucketName)) {
            // 创建bucket
            ossClient.createBucket(bucketName);
        }
    }

    /**
     * 创建存储空间
     */
    private void createBucket(String bucketName) {
        ossClient.createBucket(bucketName);
    }

    /**
     * 获取阿里云图片服务缩略压缩图 宽度固定 居中裁剪
     *
     * @param oriUrl 原图地址 不处理超过20M以上的图片
     * @param width  宽度
     * @param height 高度
     * @return 缩略图url
     */
    public String getThumbnailUrl(String oriUrl, int width, int height) {
        if (oriUrl == null || !oriUrl.toLowerCase().startsWith("http")) {
            return oriUrl;
        }
        // 等比缩放，限定在矩形框外 将图缩略成宽度为 width，高度为 height 宽度固定 居中裁剪
        StringBuilder result = new StringBuilder(oriUrl);
        result.append("?x-oss-process=image/auto-orient,1/resize,m_fill,w_");
        result.append(width);
        result.append(",h_");
        result.append(height);
        result.append("/quality,q_100");
        return result.toString();
    }

    /**
     * 获取阿里云图片服务缩略压缩图 高度固定 宽度自适应
     *
     * @param oriUrl 原图地址 不处理超过20M以上的图片
     * @param width  宽度
     * @return 缩略图url
     */
    public String getThumbnailUrlByWidth(String oriUrl, int width) {
        if (oriUrl == null || !oriUrl.toLowerCase().startsWith("http")) {
            return oriUrl;
        }
        StringBuilder result = new StringBuilder(oriUrl);
        result.append("?x-oss-process=image/auto-orient,1/resize,m_lfit,w_");
        result.append(width);
        result.append("/quality,q_100");
        return result.toString();
    }

    /**
     * 获取阿里云图片服务缩略压缩图 固定宽高 按长边缩放
     *
     * @param oriUrl 原图地址 不处理超过20M以上的图片
     * @param width  宽度
     * @param height 高度
     * @return 缩略图url
     */
    public String getThumbnailUrlByHeight(String oriUrl, int width, int height) {
        if (oriUrl == null || !oriUrl.toLowerCase().startsWith("http")) {
            return oriUrl;
        }
        StringBuilder result = new StringBuilder(oriUrl);
        result.append("?x-oss-process=image/resize,m_lfit,h_");
        result.append(height);
        result.append(",w_");
        result.append(width);
        return result.toString();
    }

    /**
     * 阿里OSS服务器多文件打包ZIP下载，批量下载
     *
     * @param key         带目录的key 多文件用,号分割
     * @param zipFileName zip包文件名
     * @param request
     * @param response
     * @return
     */
    public HttpServletResponse zipFilesDown(String key, String zipFileName, HttpServletRequest request, HttpServletResponse response) {
        try {
            String fileName = zipFileName + ".zip";
            // 创建临时文件
            File zipFile = File.createTempFile(zipFileName, ".zip");
            FileOutputStream f = new FileOutputStream(zipFile);

            CheckedOutputStream outputStream = new CheckedOutputStream(f, new Adler32());
            // 用于将数据压缩成Zip文件格式
            ZipOutputStream zos = new ZipOutputStream(outputStream);

            String[] keyList = key.split(",");
            for (String ossFile : keyList) {
                ossFile = ossFile.trim();
                // 获取Object，返回结果为OSSObject对象
                OSSObject ossObject = ossClient.getObject(bucketName, ossFile);
                // 读去Object内容  返回
                InputStream inputStream = ossObject.getObjectContent();
                // 对于每一个要被存放到压缩包的文件，都必须调用ZipOutputStream对象的putNextEntry()方法，确保压缩包里面文件不同名

                zos.putNextEntry(new ZipEntry(ossFile.split("/")[1]));
                int bytesRead = 0;
                // 向压缩文件中输出数据
                while ((bytesRead = inputStream.read()) != -1) {
                    zos.write(bytesRead);
                }
                inputStream.close();
                zos.closeEntry(); // 当前文件写完，定位为写入下一条项目
            }
            zos.close();
            String header = request.getHeader("User-Agent").toUpperCase();
            if (header.contains("MSIE") || header.contains("TRIDENT") || header.contains("EDGE")) {
                fileName = URLEncoder.encode(fileName, "utf-8");
                fileName = fileName.replace("+", "%20");    //IE下载文件名空格变+号问题
            } else {
                fileName = new String(fileName.getBytes(), "ISO8859-1");
            }
            response.reset();
            response.setContentType("text/plain");
            response.setContentType("application/octet-stream; charset=utf-8");
            response.setHeader("Location", fileName);
            response.setHeader("Cache-Control", "max-age=0");
            response.setHeader("Content-Disposition", "attachment; filename=" + fileName);

            FileInputStream fis = new FileInputStream(zipFile);
            BufferedInputStream buff = new BufferedInputStream(fis);
            BufferedOutputStream out = new BufferedOutputStream(response.getOutputStream());
            byte[] car = new byte[1024];
            int l = 0;
            while (l < zipFile.length()) {
                int j = buff.read(car, 0, 1024);
                l += j;
                out.write(car, 0, j);
            }
            // 关闭流
            fis.close();
            buff.close();
            out.close();

            ossClient.shutdown();
            // 删除临时文件
            zipFile.delete();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return response;
    }

}
