package com.onsiteservice.util;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author 潘维吉
 * @date 2019-12-13 15:08
 * @email 406798106@qq.com
 * @description 地理学相关工具类
 */
public class GeologyUtils {

    private static double EARTH_RADIUS = 6378.137;

    private static double rad(double d) {
        return d * Math.PI / 180.0;
    }

    /**
     * 根据经纬度计算两点之间的距离(米)
     */
    public static double getDistance(String lon1, String lat1, String lon2, String lat2) {
        double radLat1 = rad(Double.valueOf(lat1));
        double radLat2 = rad(Double.valueOf(lat2));
        double a = radLat1 - radLat2;
        double b = rad(Double.valueOf(lon1)) - rad(Double.valueOf(lon2));
        double s = 2 * Math.asin(Math.sqrt(Math.pow(Math.sin(a / 2), 2)
                + Math.cos(radLat1) * Math.cos(radLat2)
                * Math.pow(Math.sin(b / 2), 2)));
        s = s * EARTH_RADIUS;
        s = Math.round(s * 10000d) / 10000d;
        s = s * 1000;
        return s;
    }

    /**
     * 将坐标数组转为wkt格式字符串
     */
    private static final String WKT_POLYGON_START = "POLYGON(("; // wkt字符串开头
    private static final String WKT_POLYGON_END = "))"; // wkt字符串结尾

    public static String getWktFromPoints(List<Map<String, String>> points) {
        String pointsStr = points.stream().map(p -> p.get("lon") + " " + p.get("lat")).collect(Collectors.joining(","));
        return WKT_POLYGON_START.concat(pointsStr.concat(WKT_POLYGON_END));
    }

    public static String getWktFromPointsByArray(List<Double[]> points) {
        String pointsStr = points.stream().map(p -> p[0] + " " + p[1]).collect(Collectors.joining(","));
        return WKT_POLYGON_START.concat(pointsStr.concat(WKT_POLYGON_END));
    }

    /**
     * 将坐标数组转为wkt格式字符串
     */
    public static List<Map<String, String>> getPoints(String wkt) {
        String pointsStr = wkt.replace(WKT_POLYGON_START, "").replace(WKT_POLYGON_END, "");
        String[] points = pointsStr.split(",");
        List<Map<String, String>> pointList = new ArrayList<>(points.length);
        for (String s : points) {
            var lonLat = s.split(" ");
            pointList.add(Map.of("lon", lonLat[0], "lat", lonLat[1]));
        }
        return pointList;
    }

    public static List<Double[]> getPointsToArray(String wkt) {
        String pointsStr = wkt.replace(WKT_POLYGON_START, "").replace(WKT_POLYGON_END, "");
        String[] points = pointsStr.split(",");
        List<Double[]> pointList = new ArrayList<>(points.length);
        for (String s : points) {
            var lonLat = s.split(" ");
            pointList.add(new Double[]{Double.parseDouble(lonLat[0]), Double.parseDouble(lonLat[1])});
        }
        return pointList;
    }

    /**
     * 判断点是否在圆内
     */
    public static boolean isInCircle(String centerLon, String centerLat, String lon, String lat, Double radius) {
        // 计算两点距离
        double distance = getDistance(centerLon, centerLat, lon, lat);
        return distance <= radius;
    }

    /**
     * 判断点是否在多边形内
     *
     * @param pointLon
     * @param pointLat
     * @param pointList
     * @return
     */
    public static boolean isInPolygon(Double pointLon, Double pointLat, List<Double[]> pointList) {

        Point2D.Double point = new Point2D.Double(pointLon, pointLat);

        List<Point2D.Double> pts = new ArrayList<>();

        double polygonPoint_x = 0.0, polygonPoint_y = 0.0;

        for (Double[] pointLocation : pointList) {
            polygonPoint_x = pointLocation[0];
            polygonPoint_y = pointLocation[1];
            Point2D.Double polygonPoint = new Point2D.Double(polygonPoint_x, polygonPoint_y);
            pts.add(polygonPoint);
        }

        int N = pts.size();
        boolean boundOrVertex = true;
        int intersectCount = 0;//交叉点数量
        double precision = 2e-10; //浮点类型计算时候与0比较时候的容差
        Point2D.Double p1, p2;//临近顶点
        Point2D.Double p = point; //当前点

        p1 = pts.get(0);
        for (int i = 1; i <= N; ++i) {
            if (p.equals(p1)) {
                return boundOrVertex;
            }

            p2 = pts.get(i % N);
            if (p.x < Math.min(p1.x, p2.x) || p.x > Math.max(p1.x, p2.x)) {
                p1 = p2;
                continue;
            }

            //射线穿过算法
            if (p.x > Math.min(p1.x, p2.x) && p.x < Math.max(p1.x, p2.x)) {
                if (p.y <= Math.max(p1.y, p2.y)) {
                    if (p1.x == p2.x && p.y >= Math.min(p1.y, p2.y)) {
                        return boundOrVertex;
                    }

                    if (p1.y == p2.y) {
                        if (p1.y == p.y) {
                            return boundOrVertex;
                        } else {
                            ++intersectCount;
                        }
                    } else {
                        double xinters = (p.x - p1.x) * (p2.y - p1.y) / (p2.x - p1.x) + p1.y;
                        if (Math.abs(p.y - xinters) < precision) {
                            return boundOrVertex;
                        }

                        if (p.y < xinters) {
                            ++intersectCount;
                        }
                    }
                }
            } else {
                if (p.x == p2.x && p.y <= p2.y) {
                    Point2D.Double p3 = pts.get((i + 1) % N);
                    if (p.x >= Math.min(p1.x, p3.x) && p.x <= Math.max(p1.x, p3.x)) {
                        ++intersectCount;
                    } else {
                        intersectCount += 2;
                    }
                }
            }
            p1 = p2;
        }
        if (intersectCount % 2 == 0) { //偶数在多边形外
            return false;
        } else { //奇数在多边形内
            return true;
        }
    }


    public static void main(String[] args) {
        List<Double[]> locations = new ArrayList<>();
        locations.add(new Double[]{119.531351, 35.43546});
        locations.add(new Double[]{119.531876, 35.435435});
        locations.add(new Double[]{119.5319, 35.435917});
        locations.add(new Double[]{119.532987, 35.435922});
        locations.add(new Double[]{119.532594, 35.433276});
        locations.add(new Double[]{119.532365, 35.433202});
        locations.add(new Double[]{119.531303, 35.433281});
        locations.add(new Double[]{119.531303, 35.433281});
        locations.add(new Double[]{119.531351, 35.43546});

        double[] center = CoordinateUtils.wgs84togcj02(119.526047, 35.435047);

        Boolean isIn = isInPolygon(119.526047, 35.435047, locations);

        System.out.println(isIn);
    }

}
