package com.onsiteservice.service.order;

import com.onsiteservice.common.order.dto.DispatchServiceOrderDTO;
import com.onsiteservice.common.order.dto.FinishServiceOrderDTO;
import com.onsiteservice.common.order.dto.SendServiceOrderDTO;
import com.onsiteservice.common.order.dto.ValuationServiceOrderDTO;
import com.onsiteservice.constant.constant.BizConstants;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.constant.enums.ServiceOrderOpSourceEnum;
import com.onsiteservice.constant.enums.ServiceOrderStatusEnum;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.component.RecordComponent;
import com.onsiteservice.dao.mapper.service.*;
import com.onsiteservice.dao.mapper.sys.SysUserMapper;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.order.ServiceOrder;
import com.onsiteservice.entity.service.ServiceValuator;
import com.onsiteservice.entity.service.ServiceValuatorAssign;
import com.onsiteservice.entity.service.ServiceWorker;
import com.onsiteservice.entity.service.ServiceWorkerAssign;
import com.onsiteservice.entity.sys.SysUser;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.util.aliyun.SmsUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class ServiceOrderService extends AbstractMapper<ServiceOrder> {


    /**
     * 短信相关
     */
    @Value("${aliyun.sms.sign}")
    private String sign;
    @Autowired
    private SmsUtils smsUtils;

    @Resource
    private ServiceOrderMapper serviceOrderMapper;

    @Resource
    private ServiceWorkerMapper serviceWorkerMapper;
    @Resource
    private ServiceWorkerAssignMapper serviceWorkerAssignMapper;

    @Resource
    private ServiceValuatorMapper serviceValuatorMapper;
    @Resource
    private ServiceValuatorAssignMapper serviceValuatorAssignMapper;

    @Resource
    private UserMapper userMapper;

    @Autowired
    private RecordComponent recordComponent;

    @Resource
    private SysUserMapper sysUserMapper;


    /**
     * 指派估价员 管理员操作
     */
    public int valuation(ValuationServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("order valuation dto: {}, userId: {}", dto, userId);

        WorkUser user = checkAuth(userId, BizCodeEnum.NO_AUTH_VALUATION_ORDER, sourceEnum);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.RESERVE.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_VALUATION);
        }

        User valuatorUser = userMapper.selectByPrimaryKey(dto.getHostId());
        if (Objects.isNull(valuatorUser)) {
            throw new ServiceException(BizCodeEnum.SERVICE_VALUATOR_NOT_EXIST);
        }

        serviceValuatorAssignMapper.insertSelective(buildServiceValuatorAssign(dto, userId));

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.VALUATION.getStatus());
        serviceOrder.setModifyBy(userId);
        serviceOrder.setValuatorId(valuatorUser.getId());

        int result = this.updateByPrimaryKeySelective(serviceOrder);

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.VALUATION.getMsg(), user.getName(), valuatorUser.getUserName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.VALUATION.getStatus(), description,
                    sourceEnum, valuatorUser.getId(), dto.getRemark(), dto.getExpectArrivalTime());
        }

        return result;
    }


    public int revaluation(ValuationServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("order revaluation dto: {}, userId: {}", dto, userId);

        WorkUser user = checkAuth(userId, BizCodeEnum.NO_AUTH_REVALUATION_ORDER, sourceEnum);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.VALUATION.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_REVALUATION);
        }

        User valuatorUser = userMapper.selectByPrimaryKey(dto.getHostId());
        if (Objects.isNull(valuatorUser)) {
            throw new ServiceException(BizCodeEnum.SERVICE_VALUATOR_NOT_EXIST);
        }

        disableServiceValuatorAssign(serviceOrder, userId, false);

        serviceValuatorAssignMapper.insertSelective(buildServiceValuatorAssign(dto, userId));

        serviceOrder.setValuatorId(dto.getHostId());
        int result = this.updateByPrimaryKeySelective(serviceOrder);

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.REVALUATION.getMsg(), user.getName(), valuatorUser.getUserName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.REVALUATION.getStatus(), description,
                    sourceEnum, valuatorUser.getId(), dto.getRemark(), dto.getExpectArrivalTime());
        }

        return result;
    }


    /**
     * 发送订单 管理员操作
     */
    public int sendOrder(SendServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("sendOrder dto: {}, userId: {}", dto, userId);

        // 判断发送者权限 只有客服和估价员可以发送订单
        WorkUser user = checkAuth(userId, BizCodeEnum.NO_AUTH_SEND_ORDER, sourceEnum);

        // 判断能发送的状态: 估价可以发送,下单可以直接发送
        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.VALUATION.getStatus()) && !serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.RESERVE.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_SEND);
        }

        disableServiceValuatorAssign(serviceOrder, userId, null);

        // 估价价格
        serviceOrder.setPrice(dto.getPrice());
        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.SEND.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        try {
            // 发短信
            log.info("sendOrder send msg to phone: {}", serviceOrder.getPhone());
            // TODO 短信通知客户完成支付
            smsUtils.send(sign, "", "", serviceOrder.getPhone());
        } catch (Exception e) {
            log.error("sendOrder error, ", e);
        }

        // TODO 小程序通知客户完成支付

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.SEND.getMsg(), (user.getIsAdmin() ? ServiceUserTypeEnum.ADMIN.getName() : ServiceUserTypeEnum.VALUATOR.getName()) + user.getName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.SEND.getStatus(), description, sourceEnum, null, null, null);
        }

        return result;
    }


    public int dispatch(DispatchServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("order dispatch dto: {}, userId: {}", dto, userId);

        WorkUser user = checkAuth(userId, BizCodeEnum.NO_AUTH_DISPATCH_ORDER, sourceEnum);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.PAY.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_DISPATCH);
        }

        if (System.currentTimeMillis() >= serviceOrder.getExpectArrivalTime().getTime()) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_TIMEOUT_CANNOT_DISPATCH);
        }

        ServiceWorker serviceWorker = serviceWorkerMapper.selectByPrimaryKey(dto.getHostId());
        if (Objects.isNull(serviceWorker)) {
            throw new ServiceException(BizCodeEnum.SERVICE_WORKER_NOT_EXIST);
        }

        serviceWorkerAssignMapper.insertSelective(buildServiceWorkerAssign(dto, serviceOrder, userId));

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.DISPATCH.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        // TODO 短信 "您的订单已为您指派维修工,请保持手机畅通!"

        // TODO 小程序通知 您的订单已为您指派维修工~~~

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.DISPATCH.getMsg(), user.getName(), serviceWorker.getName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.DISPATCH.getStatus(), description, sourceEnum, serviceWorker.getId(), dto.getRemark(), null);
        }

        return result;
    }


    public int redispatch(DispatchServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("order redispatch dto: {}, userId: {}", dto, userId);

        WorkUser user = checkAuth(userId, BizCodeEnum.NO_AUTH_REDISPATCH_ORDER, sourceEnum);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.DISPATCH.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_REDISPATCH);
        }

        if (System.currentTimeMillis() >= serviceOrder.getExpectArrivalTime().getTime()) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_TIMEOUT_CANNOT_DISPATCH);
        }

        ServiceWorker serviceWorker = serviceWorkerMapper.selectByPrimaryKey(dto.getHostId());
        if (Objects.isNull(serviceWorker)) {
            throw new ServiceException(BizCodeEnum.SERVICE_WORKER_NOT_EXIST);
        }

        disableServiceWorkerAssign(serviceOrder, userId, false);

        int result = serviceWorkerAssignMapper.insertSelective(buildServiceWorkerAssign(dto, serviceOrder, userId));

        // TODO 短信 "您的订单已为您重新指派维修工,请保持手机畅通!"

        // TODO 小程序通知 您的订单已为您重新指派维修工~~~

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.REDISPATCH.getMsg(), user.getName(), serviceWorker.getName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.REDISPATCH.getStatus(), description, sourceEnum, serviceWorker.getId(), dto.getRemark(), null);
        }

        return result;
    }


    /**
     * 完成
     */
    public int finishOrder(FinishServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("finishOrder dto: {}, userId: {}", dto, userId);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.DISPATCH.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_FINISH);
        }

        WorkUser user;
        if (sourceEnum == ServiceOrderOpSourceEnum.MINI_APP) {
            var miniUser = userMapper.selectByPrimaryKey(userId);
            boolean isUser = serviceOrder.getAccountNo().equals(userId);
            boolean isAdmin = !Objects.isNull(miniUser) && miniUser.getRoleType().equals(ServiceUserTypeEnum.ADMIN.getId());
            if (!isUser && !isAdmin) {
                throw new ServiceException(BizCodeEnum.NO_AUTH_FINISH_ORDER);
            }
            user = new WorkUser(miniUser.getId(), miniUser.getUserName(), isAdmin);
        } else {
            var sysUser = sysUserMapper.selectByPrimaryKey(userId);
            user = new WorkUser(sysUser.getId(), sysUser.getRealName(), true);
        }


        disableServiceWorkerAssign(serviceOrder, userId, null);

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.FINISH.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.FINISH.getMsg(), user.getIsAdmin() ? ServiceUserTypeEnum.ADMIN.getName() + user.getName() : ServiceUserTypeEnum.USER.getName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.FINISH.getStatus(), description, sourceEnum, null, null, null);
        }

        return result;
    }


    public ServiceOrder checkOrder(Long id, Long userId) {
        Condition c = new Condition(ServiceOrder.class);
        c.createCriteria().andEqualTo("id", id);
        if (userId != null) {
            c.createCriteria().andEqualTo(BizConstants.UserConstants.ACCOUNT_NO, userId);
        }

        List<ServiceOrder> serviceOrderList = serviceOrderMapper.selectByCondition(c);
        if (CollectionUtils.isEmpty(serviceOrderList)) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_NOT_EXIST);
        }

        return serviceOrderList.get(0);
    }


    private WorkUser checkAuth(Long userId, BizCodeEnum biz, ServiceOrderOpSourceEnum source) {
        if (source == ServiceOrderOpSourceEnum.MINI_APP) {
            User user = userMapper.selectByPrimaryKey(userId);
            if (Objects.isNull(user) || !user.getRoleType().equals(ServiceUserTypeEnum.ADMIN.getId())) {
                throw new ServiceException(biz);
            }
            return new WorkUser(user.getId(), user.getUserName(), user.getRoleType().equals(ServiceUserTypeEnum.ADMIN.getId()));
        }
        SysUser sysUser = sysUserMapper.selectByPrimaryKey(userId);
        if (Objects.isNull(sysUser)) {
            throw new ServiceException(biz);
        }
        return new WorkUser(sysUser.getId(), sysUser.getRealName(), true);
    }


    private ServiceValuatorAssign buildServiceValuatorAssign(ValuationServiceOrderDTO dto, Long userId) {
        ServiceValuatorAssign serviceValuatorAssign = new ServiceValuatorAssign();
        // 估价员id
        serviceValuatorAssign.setValuatorId(dto.getHostId());
        // 订单id
        serviceValuatorAssign.setOrderId(dto.getId());
        // 估价员上门时间
        serviceValuatorAssign.setAssignTime(dto.getExpectArrivalTime());
        // 创建人
        serviceValuatorAssign.setCreateBy(userId);

        return serviceValuatorAssign;
    }

    private ServiceWorkerAssign buildServiceWorkerAssign(DispatchServiceOrderDTO dto, ServiceOrder serviceOrder, Long userId) {
        ServiceWorkerAssign serviceWorkerAssign = new ServiceWorkerAssign();
        // 维修工id
        serviceWorkerAssign.setWorkerId(dto.getHostId());
        // 订单id
        serviceWorkerAssign.setOrderId(dto.getId());
        // 维修工上门时间
        serviceWorkerAssign.setAssignTime(serviceOrder.getExpectArrivalTime());
        // 创建人
        serviceWorkerAssign.setCreateBy(userId);

        return serviceWorkerAssign;
    }


    /**
     * 估价员上门时间失效
     *
     * @param serviceOrder  订单
     * @param userId        用户id
     * @param finalExecuted 逻辑上谁最有可能是最终的上门人员
     */
    public void disableServiceValuatorAssign(ServiceOrder serviceOrder, Long userId, Boolean finalExecuted) {
        log.info("disableServiceValuatorAssign serviceOrder: {}, userId: {}, finalExecuted: {}", serviceOrder, userId, finalExecuted);

        ServiceValuatorAssign serviceValuatorAssign = new ServiceValuatorAssign();
        serviceValuatorAssign.setModifyBy(userId);
        serviceValuatorAssign.setDeleted(true);
        if (finalExecuted != null) {
            serviceValuatorAssign.setFinalExecuted(finalExecuted);
        }

        Condition c = new Condition(ServiceValuatorAssign.class);
        c.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, serviceOrder.getId());

        serviceValuatorAssignMapper.updateByConditionSelective(serviceValuatorAssign, c);
    }

    /**
     * 维修工上门时间失效
     *
     * @param serviceOrder 订单
     * @param userId       用户id
     */
    public void disableServiceWorkerAssign(ServiceOrder serviceOrder, Long userId, Boolean finalExecuted) {
        log.info("disableServiceWorkerAssign serviceOrder: {}, userId: {}, finalExecuted: {}", serviceOrder, userId, finalExecuted);

        ServiceWorkerAssign serviceWorkerAssign = new ServiceWorkerAssign();
        serviceWorkerAssign.setModifyBy(userId);
        serviceWorkerAssign.setDeleted(true);
        if (finalExecuted != null) {
            serviceWorkerAssign.setFinalExecuted(finalExecuted);
        }

        Condition c = new Condition(ServiceWorkerAssign.class);
        c.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, serviceOrder.getId());

        serviceWorkerAssignMapper.updateByConditionSelective(serviceWorkerAssign, c);
    }


}
