package com.onsiteservice.admin.security;


import com.onsiteservice.admin.security.authentication.CustomUserDetailsService;
import com.onsiteservice.admin.security.authentication.JwtAuthenticationFilter;
import com.onsiteservice.admin.security.authentication.JwtLoginFilter;
import com.onsiteservice.admin.security.handler.CustomAccessDeniedHandler;
import com.onsiteservice.admin.security.handler.CustomAuthenticationEntryPoint;
import com.onsiteservice.constant.constant.Constants;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpMethod;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.config.BeanIds;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.access.AccessDeniedHandler;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;

import javax.annotation.Resource;

/**
 * 安全权限配置
 */
@Order(2) // 核心core安全配置优先级
@ConditionalOnProperty(prefix = "project.admin-security", name = {"enabled"}, matchIfMissing = true)
@Configuration
@EnableWebSecurity(debug = true)
public class SecurityConfiguration extends WebSecurityConfigurerAdapter {

    @Value("${project.jwt.ant-paths}")
    private String antPaths;

    @Resource
    private CustomAuthenticationEntryPoint customAuthenticationEntryPoint;

    public static void main(String[] args) {
        System.out.println(new BCryptPasswordEncoder().encode("123456"));
    }

    /**
     * 用于配置需要拦截的url路径、jwt过滤器及出异常后的处理器
     */
    @Override
    protected void configure(HttpSecurity http) throws Exception {
        String loginPath = Constants.REQUEST_PREFIX + "/login";
        http.cors().and()           // 开启cors跨域
                .csrf().disable()   // 禁用csrf 由于使用的是JWT，我们这里不需要csrf
                .exceptionHandling()
                .authenticationEntryPoint(customAuthenticationEntryPoint) // 自定义401认证异常响应数据 用来解决匿名用户访问无权限资源时的异常
                .accessDeniedHandler(accessDeniedHandler()).and() // 自定义403授权异常响应数据 拒绝访问
                .sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS).and() // 基于token 不需要session
                .authorizeRequests() // 限定签名成功的请求
                .antMatchers(HttpMethod.OPTIONS).permitAll() // 跨域请求会先进行一次options请求
                .antMatchers(antPaths.split(",")).permitAll() // 放行不需要认证的url
                .antMatchers(HttpMethod.POST, loginPath).permitAll() // 允许POST类型登录接口
                //.antMatchers("/**").permitAll() // 测试时全部运行访问
                .anyRequest().authenticated() // 除上面外的所有请求全部需要鉴权认证
                .and()
                // 添加*/login登录请求过滤器
                .addFilterBefore(new JwtLoginFilter(loginPath, authenticationManager()), UsernamePasswordAuthenticationFilter.class)
                // 过滤其他请求以检查JWT在header中的存在
                .addFilterBefore(new JwtAuthenticationFilter(), UsernamePasswordAuthenticationFilter.class);

    }

    @Bean
    public UserDetailsService customUserDetailsService() {
        return new CustomUserDetailsService();
    }

    /**
     * 用于配置UserDetailsService及PasswordEncode
     */
    @Override
    public void configure(AuthenticationManagerBuilder auth) throws Exception {
        auth.userDetailsService(customUserDetailsService())
                .passwordEncoder(passwordEncoder());//对密码的加密处理，如果user中密码没有加密，则可以不加此方法。注意加密请使用security自带的加密方式。
    }

    @Bean
    public PasswordEncoder passwordEncoder() {
        return new BCryptPasswordEncoder();
    }

    @Bean(name = BeanIds.AUTHENTICATION_MANAGER)
    @Override
    public AuthenticationManager authenticationManagerBean() throws Exception {
        return super.authenticationManagerBean();
    }

    @Bean
    public AccessDeniedHandler accessDeniedHandler() {
        return new CustomAccessDeniedHandler();
    }

}
