package com.onsiteservice.admin.controller.order;

import com.onsiteservice.admin.controller.order.dto.OrderPageDTO;
import com.onsiteservice.admin.controller.order.vo.OrderDetailVO;
import com.onsiteservice.admin.controller.order.vo.OrderPageVO;
import com.onsiteservice.admin.service.order.AdminOrderService;
import com.onsiteservice.common.annotation.dict.Dict;
import com.onsiteservice.common.annotation.user.CurrentUserId;
import com.onsiteservice.common.order.dto.DispatchServiceOrderDTO;
import com.onsiteservice.common.order.dto.FinishServiceOrderDTO;
import com.onsiteservice.common.order.dto.SendServiceOrderDTO;
import com.onsiteservice.common.order.dto.ValuationServiceOrderDTO;
import com.onsiteservice.constant.enums.ServiceOrderOpSourceEnum;
import com.onsiteservice.entity.order.ServiceOrder;

import com.onsiteservice.core.result.Result;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.service.order.ServiceOrderService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.NonNull;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import springfox.documentation.annotations.ApiIgnore;

import javax.annotation.Resource;
import javax.validation.constraints.Positive;

import static com.onsiteservice.core.result.ResultGenerator.fail;
import static com.onsiteservice.core.result.ResultGenerator.success;

/**
 * @author 商天翔
 * @date 2022-07-14 15:11
 * @description ServiceOrderController控制类
 */
@ApiIgnore
@Api(tags = "ServiceOrderController")
@RestController
@RequestMapping("/order")
@Validated
public class ServiceOrderController {

    @Resource
    private AdminOrderService adminOrderService;
    @Resource
    private ServiceOrderService serviceOrderService;

    @Dict(code = "BIZ0002", name = "orderStatus")
    @ApiOperation(value = "分页查询列表", notes = "作者: 商天翔")
    @PostMapping("/page")
    public Result<PageInfoVO<OrderPageVO>> getPage(@RequestBody @NonNull @Validated OrderPageDTO param) {
        return success(adminOrderService.getPage(param), "获取分页列表");
    }

    @ApiOperation(value = "根据id查询", notes = "作者: 商天翔")
    @GetMapping("/{id}")
    public Result<OrderDetailVO> getDetails(@PathVariable @Positive Long id) {
        return success(adminOrderService.getOrderDetail(id));
    }

    @ApiOperation(value = "订单估价")
    @PostMapping("valuation")
    public Result valuation(@RequestBody @NonNull @Validated ValuationServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.valuation(dto, userId, ServiceOrderOpSourceEnum.WEB) == 1 ? success() : fail("估价失败");
    }

    @ApiOperation(value = "修改估价")
    @PostMapping("revaluation")
    public Result revaluation(@RequestBody @NonNull @Validated ValuationServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.revaluation(dto, userId, ServiceOrderOpSourceEnum.WEB) == 1 ? success() : fail("修改估价失败");
    }

    /**
     * 发送订单
     */
    @ApiOperation(value = "发送订单")
    @PostMapping("send")
    public Result sendPrice(@RequestBody @NonNull @Validated SendServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.sendOrder(dto, userId, ServiceOrderOpSourceEnum.WEB) == 1 ? success() : fail("发送失败");
    }

    @ApiOperation(value = "派单")
    @PostMapping("dispatch")
    public Result dispatch(@RequestBody @NonNull @Validated DispatchServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.dispatch(dto, userId, ServiceOrderOpSourceEnum.WEB) == 1 ? success() : fail("派单失败");
    }

    @ApiOperation(value = "修改派单")
    @PostMapping("redispatch")
    public Result redispatch(@RequestBody @NonNull @Validated DispatchServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.redispatch(dto, userId, ServiceOrderOpSourceEnum.WEB) == 1 ? success() : fail("修改派单失败");
    }

    /**
     * 未派单之前不能完成,已申请退款的也可以点击完成
     */
    @ApiOperation(value = "完成订单")
    @PostMapping("finish")
    public Result finish(@RequestBody @NonNull @Validated FinishServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.finishOrder(dto, userId, ServiceOrderOpSourceEnum.WEB) == 1 ? success() : fail("取消失败");
    }

}
