package com.onsiteservice.dao.common.schedule;

import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Lazy;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;

import javax.annotation.Resource;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * @author 潘维吉
 * @date 2019-02-27 11:17
 * 定时任务配置 多线程并行任务
 * SpringBoot默认执行方式是单线程串行执行，无论有多少task，都是一个线程串行执行，多线程并行任务需手动配置
 */
@ConditionalOnProperty(prefix = "project.schedule", name = {"enabled"}, matchIfMissing = false)
@Configuration
@EnableScheduling
@Lazy(false)
public class ScheduleConfig implements SchedulingConfigurer {

    /** 定义计划线程池注入 */
    @Resource
    private TaskScheduler myTaskScheduler;

    /**
     * 配置任务
     */
    @Override
    public void configureTasks(ScheduledTaskRegistrar scheduledTaskRegistrar) {
        //简单粗暴的方式直接指定
        //scheduledTaskRegistrar.setScheduler(Executors.newScheduledThreadPool(5));
        //可以自定义的线程池，方便线程的使用与维护
        scheduledTaskRegistrar.setTaskScheduler(myTaskScheduler);

    }

    /**
     * 自定义任务线程池
     */
    @Bean(name = "myTaskScheduler")
    public TaskScheduler getMyThreadPoolTaskScheduler() {
        ThreadPoolTaskScheduler taskScheduler = new ThreadPoolTaskScheduler();
        // 定时任务执行线程池核心线程数
        taskScheduler.setPoolSize(10);
        taskScheduler.setRemoveOnCancelPolicy(true);
        taskScheduler.setThreadNamePrefix("MyTaskScheduler-");
        taskScheduler.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
        //调度器shutdown被调用时等待当前被调度的任务完成
        taskScheduler.setWaitForTasksToCompleteOnShutdown(true);
        //等待时长
        taskScheduler.setAwaitTerminationSeconds(60);
        return taskScheduler;
    }


}
