package com.onsiteservice.core.security.jwt;

import com.alibaba.fastjson.JSON;
import com.onsiteservice.constant.constant.Constants;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.ResultCodeEnum;
import io.jsonwebtoken.Claims;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author 潘维吉
 * @date 2018-05-20
 * JWT自定义拦截器，判断此次请求是否有token权限
 * <p>
 * preHandle在业务处理器处理请求之前被调用
 * postHandle在业务处理器处理请求执行完成后,生成视图之前执行
 * afterCompletion在DispatcherServlet完全处理完请求后即在视图渲染完成后回调
 */
@Component
@Slf4j
public class JwtInterceptor implements HandlerInterceptor {

    /** 是否开启jwt认证 默认true开启 */
    @Value("${project.jwt.enabled:true}")
    private boolean jwtEnabled;

    /**
     * 请求前预处理
     * return false 则拦截中断
     */
    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        if (!jwtEnabled) {
            return true;
        }
        // 拦截获取http请求的URI地址和方法
        String uri = request.getRequestURI();

        // 不处理认证  如果不是映射到方法直接通过  自定义不认证路径直接通过  HandlerMethod及子类主要用于封装方法调用相关信息
        if (!(handler instanceof HandlerMethod) || !JwtPathProperties.isAuth(uri)) {
            return true;
        }

/*        if (!uri.startsWith(Constants.REQUEST_PREFIX)) {
            throw new ServiceException("请求路径错误, 请包含URL前缀 " + Constants.REQUEST_PREFIX);
        }*/

        String authorizationHeader = JwtManager.AUTHORIZATION_HEADER;
        String bearer = JwtManager.BEARER;
        // 从header中得到token
        String authorization = request.getHeader(authorizationHeader);
        // 返回401错误
        if (authorization == null) {
            throw new ServiceException(ResultCodeEnum.UNAUTHORIZED.getCode(),
                    "Header认证" + authorizationHeader + "信息为空");
        }

        try {
            // 解析验证token
            Claims claims = JwtManager.parseToken(authorization.replaceAll(bearer, ""));
            if (claims != null) {
                // 如果token验证成功，将token对应的用户id
                Object userId = JSON.parseObject(claims.getSubject()).get(JwtManager.USER_ID);
                // 如果token验证成功，将token对应的用户id存在request中，便于之后注入
                request.setAttribute(Constants.CURRENT_USER_ID, userId);
                //log.info("JwtInterceptor当前请求用户id=" + userId.toString());
                // 基础token路径验证 基础token只能访问未登录之前指定的path路径
                if (Long.valueOf(userId.toString()) <= 0) {
                    Boolean isCanAccess = JwtPathProperties.isBaseTokenCanAccess(uri);
                    if (!isCanAccess) {
                        throw new ServiceException(ResultCodeEnum.UNAUTHORIZED.getCode(), "基础token无法访问" + uri + "路径");
                    }
                }
                try {
                    // 适配 Saas化应用多公司id
                    Object companyId = JSON.parseObject(claims.getSubject()).get(JwtManager.COMPANY_ID);
                    request.setAttribute(Constants.CURRENT_COMPANY_ID, companyId);
                } catch (Exception e) {
                }
                return true;
            }
        } catch (Exception e) {
            // 如果验证token失败，并且方法注明了Authorization，返回401错误
            throw new ServiceException(ResultCodeEnum.UNAUTHORIZED.getCode(),
                    (e instanceof ServiceException) ? e.getMessage() : "验证token认证失败或已过期");
        }
        // claims为null情况
        throw new ServiceException(ResultCodeEnum.UNAUTHORIZED.getCode(), "token解析异常");
    }
}
