package com.onsiteservice.core.config;

import org.apache.catalina.connector.Connector;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.web.embedded.tomcat.TomcatConnectorCustomizer;
import org.springframework.boot.web.embedded.tomcat.TomcatServletWebServerFactory;
import org.springframework.boot.web.servlet.MultipartConfigFactory;
import org.springframework.boot.web.servlet.server.ServletWebServerFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.unit.DataSize;

import javax.servlet.MultipartConfigElement;

/**
 * @author 潘维吉
 * @date 2018-05-27
 * Tomcat配置类
 * 内嵌Tomcat服务器 默认http连接  最大连接数和最大线程数  上传文件大小限制等设置
 */
@ConditionalOnProperty(prefix = "project.tomcat", name = {"enabled"}, matchIfMissing = true)
@Configuration
public class TomcatConfig {

    @Value("${server.tomcat.max-file-size:3000MB}")
    private String maxFileSize;
    @Value("${server.tomcat.max-request-size:5000MB}")
    private String maxRequestSize;

    @Bean
    public ServletWebServerFactory servletContainer() {
        TomcatServletWebServerFactory tomcat = new TomcatServletWebServerFactory();
        tomcat.addConnectorCustomizers(new CustomTomcatConnectorCustomizer());
        return tomcat;
    }

    /**
     * 上传文件大小限制设置
     * DataSize是 Spring Boot 2.1.0以后的使用方式
     */
    @Bean
    public MultipartConfigElement multipartConfigElement() {
        MultipartConfigFactory factory = new MultipartConfigFactory();
        // 最大允许单次上传文件大小 单个数据大小
        factory.setMaxFileSize(DataSize.parse(maxFileSize));
        // 最大允许单次multipart/form-data数据请求大小  总上传数据大小
        factory.setMaxRequestSize(DataSize.parse(maxRequestSize));
        return factory.createMultipartConfig();
    }

    /**
     * 自定义Tomcat参数配置
     */
    public class CustomTomcatConnectorCustomizer implements TomcatConnectorCustomizer {

        public CustomTomcatConnectorCustomizer() {
        }

        @Override
        public void customize(Connector connector) {
            //connector.setAttribute("connectionTimeout", 20000); //连接超时时间 连接无数据后20秒就断开
            //connector.setAttribute("acceptorThreadCount", 2); //线程数 CPU数 参考cpu核数
            //connector.setAttribute("minSpareThreads", 5); //最小初始化线程
            //connector.setAttribute("maxSpareThreads", 50); //最大初始化线程
            //connector.setAttribute("maxThreads", 1000); //最大线程数 最大并发数
            //connector.setAttribute("maxConnections", 1000); //最大连接数
            //connector.setAttribute("compression", "on"); //开启压缩 助于减少客户端下载，提高客户端响应速度 压缩是对CPU的消耗 compressionMinSize配置是大于一定大小的响应内容的情况下才开启压缩
            //connector.setAttribute("protocol", "org.apache.coyote.http11.Http11Nio2Protocol"); //http协议
            //connector.setAttribute("redirectPort", 443); //重定向端口
        }
    }

}
