import type { BaseModel, PageQuery } from '#/api/baseModel';

import { commonExport } from '#/api/helper';
import { requestClient } from '#/api/request';

export namespace SysTenantApi {
  export interface Tenant extends BaseModel {
    /**
     * 租户编号
     */
    tenantCode?: string;
    /**
     * 联系人
     */
    contactUserName?: string;
    /**
     * 联系电话
     */
    contactPhone?: string;
    /**
     * 企业名称
     */
    companyName?: string;
    /**
     * 统一社会信用代码
     */
    licenseNumber?: string;
    /**
     * 地址
     */
    address?: string;
    /**
     * 域名
     */
    domain?: string;
    /**
     * 企业简介
     */
    intro?: string;
    /**
     * 租户套餐ID
     */
    packageId?: number;

    /**
     * 过期时间
     */
    expireTime?: string;
    /**
     * 用户数量（-1不限制）
     */
    accountCount?: number;

    /**
     * 租户状态（1正常 0停用）
     */
    status?: number;

    /**
     * 备注
     */
    remark?: string;
  }
}

/**
 * 查询租户列表
 * @param params
 * @returns {*} page
 */
export function apiPage(params: PageQuery) {
  return requestClient.get('/system/tenant/page', { params });
}
/**
 * 查询租户详细
 * @param id
 */
export function apiDetail(id: number) {
  return requestClient.get(`/system/tenant/${id}`);
}
/**
 * 新增租户
 * @param data
 */
export function apiAdd(data: SysTenantApi.Tenant) {
  return requestClient.post('/system/tenant', data);
}
/**
 * 修改租户
 * @param data
 */
export function apiUpdate(data: SysTenantApi.Tenant) {
  return requestClient.put('/system/tenant', data);
}
/**
 * 删除租户
 * @param id
 */
export function apiDelete(id: Array<number> | number) {
  return requestClient.delete(`/system/tenant/${id}`);
}
/**
 * 导出租户
 * @param params
 */
export function apiExport(params: PageQuery) {
  return commonExport('/system/tenant/export', params);
}
/**
 * 修改租户套餐状态
 * @param data
 */
export function apiChangeStatus(data: SysTenantApi.Tenant) {
  return requestClient.put('/system/tenant/changeStatus', data);
}

export function apiTenantDynamicClear() {
  return requestClient.get('/system/tenant/dynamic/clear');
}

export function apiTenantDynamicToggle(id: number) {
  return requestClient.get(`/system/tenant/dynamic/${id}`);
}

/**
 * 租户套餐同步
 * @param tenantId 租户id
 * @param packageId 套餐id
 * @returns void
 */
export function tenantSyncPackage(tenantId: number, packageId: number) {
  return requestClient.get('/system/tenant/syncTenantPackage', {
    params: { packageId, tenantId },
    successMessageMode: 'message',
  });
}

/**
 * 同步租户字典
 * @param tenantId 租户ID
 * @returns void
 */
export function tenantSyncDict(tenantId?: string) {
  return requestClient.get('/system/tenant/syncTenantDict', {
    params: { tenantId },
    successMessageMode: 'message',
  });
}
