import type { LeaveForm, LeaveQuery, LeaveVO } from './model';

import type { ID, IDS, PageResult } from '#/api/baseModel';

import { commonExport } from '#/api/helper';
import { requestClient } from '#/api/request';

/**
 * 查询请假申请列表
 * @param params
 * @returns 请假申请列表
 */
export function leaveList(params?: LeaveQuery) {
  return requestClient.get<PageResult<LeaveVO>>('/workflow/leave/list', {
    params,
  });
}

/**
 * 导出请假申请列表
 * @param params
 * @returns 请假申请列表
 */
export function leaveExport(params?: LeaveQuery) {
  return commonExport('/workflow/leave/export', params ?? {});
}

/**
 * 查询请假申请详情
 * @param id id
 * @returns 请假申请详情
 */
export function leaveInfo(id: ID) {
  return requestClient.get<LeaveVO>(`/workflow/leave/${id}`);
}

/**
 * 新增请假申请
 * @param data
 * @returns void
 */
export function leaveAdd(data: LeaveForm) {
  return requestClient.post<LeaveVO>('/workflow/leave', data, {
    successMessageMode: 'message',
  });
}

/**
 * 更新请假申请
 * @param data
 * @returns void
 */
export function leaveUpdate(data: LeaveForm) {
  return requestClient.put<LeaveVO>('/workflow/leave', data, {
    successMessageMode: 'message',
  });
}

/**
 * 删除请假申请
 * @param id id
 * @returns void
 */
export function leaveRemove(id: ID | IDS) {
  return requestClient.delete(`/workflow/leave/${id}`, {
    successMessageMode: 'message',
  });
}

/**
 * 提交 & 发起流程(后端发起)
 * @param data data
 * @returns void
 */
export function submitAndStartWorkflow(data: LeaveForm) {
  return requestClient.post('/workflow/leave/submitAndFlowStart', data, {
    successMessageMode: 'message',
  });
}
