import type { VbenFormSchema } from '#/adapter/form';
import type { VxeTableGridOptions } from '#/adapter/vxe-table';

import { markRaw } from 'vue';

import { useAccess } from '@vben/access';
import { DictEnum } from '@vben/constants';
import { getVxePopupContainer } from '@vben/utils';

import { Tag } from 'ant-design-vue';

import { changeRoleStatus } from '#/api/auth/role';
import { DefaultSlot } from '#/components/global/slot';
import { TreeSelectPanel } from '#/components/tree';
import { getDictOptions } from '#/utils/dict';

/**
 * authScopeOptions user也会用到
 */
export const authScopeOptions = [
  { color: 'green', label: '全部数据权限', value: 1 },
  { color: 'orange', label: '本部门数据权限', value: 2 },
  { color: 'cyan', label: '本部门及以下数据权限', value: 3 },
  { color: 'error', label: '仅本人数据权限', value: 4 },
  { color: 'default', label: '自定义数据权限', value: 5 },
];

const { hasAccessByCodes } = useAccess();

export const querySchema: VbenFormSchema[] = [
  {
    component: 'Input',
    fieldName: 'name',
    label: '角色名称',
  },
  {
    component: 'Input',
    fieldName: 'roleKey',
    label: '权限字符',
  },
  {
    component: 'Select',
    componentProps: {
      options: getDictOptions(DictEnum.SYS_COMMON_STATUS, true),
    },
    fieldName: 'status',
    label: '状态',
  },
  {
    component: 'Input',
    fieldName: 'dataScope',
    label: '数据范围',
  },
];

export function useColumns(): VxeTableGridOptions['columns'] {
  return [
    {
      title: '角色名称',
      field: 'name',
    },
    {
      title: '权限字符',
      field: 'roleKey',
      slots: {
        default: ({ row }) => {
          return <Tag color="processing">{row.roleKey}</Tag>;
        },
      },
    },
    {
      title: '数据范围',
      field: 'dataScope',
      cellRender: {
        name: 'CellTag',
        options: authScopeOptions,
      },
    },
    {
      title: '状态',
      field: 'status',
      cellRender: {
        name: 'CellSwitch',
        props: {
          api: changeRoleStatus,
          confirm: true,
          disabled: () => {
            return !hasAccessByCodes(['auth:role:edit']);
          },
        },
      },
    },
    {
      title: '备注',
      field: 'remark',
      width: 200,
    },
    {
      align: 'right',
      slots: { default: 'action' },
      // cellRender: {
      //   attrs: {
      //     nameField: 'name',
      //     nameTitle: '角色',
      //     onClick: onActionClick,
      //   },
      //   name: 'CellOperation',
      //   options: [
      //     {
      //       code: 'dataScope',
      //       text: '数据权限',
      //       disabled: (row: SysRoleApi.Role) => {
      //         return !!(row.id === 1);
      //       },
      //     },
      //     'edit', // 默认的编辑按钮
      //     'delete', // 默认的删除按钮
      //   ],
      // },
      field: 'action',
      fixed: 'right',
      headerAlign: 'center',
      resizable: false,
      showOverflow: false,
      title: '操作',
      width: 'auto',
    },
  ];
}

export const dataScopeModalSchemas: VbenFormSchema[] = [
  {
    component: 'Input',
    dependencies: {
      show: () => false,
      triggerFields: [''],
    },
    fieldName: 'id',
    label: '角色ID',
  },
  {
    component: 'Input',
    componentProps: {
      disabled: true,
    },
    fieldName: 'name',
    label: '角色名称',
  },
  {
    component: 'Input',
    componentProps: {
      disabled: true,
    },
    fieldName: 'roleKey',
    label: '权限标识',
  },
  {
    component: 'Select',
    componentProps: {
      allowClear: false,
      getVxePopupContainer,
      options: authScopeOptions,
    },
    fieldName: 'dataScope',
    help: '更改后需要用户重新登录才能生效',
    label: '权限范围',
  },
  {
    component: 'Radio',
    dependencies: {
      show: () => false,
      triggerFields: [''],
    },
    fieldName: 'deptCheckStrictly',
    label: 'deptCheckStrictly',
  },
  {
    // 这种的场景基本上是一个组件需要绑定两个或以上的场景
    component: markRaw(DefaultSlot),
    defaultValue: [],
    componentProps: {
      rootDivAttrs: {
        class: 'w-full',
      },
    },
    dependencies: {
      show: (values) => values.dataScope === 5,
      triggerFields: ['dataScope'],
    },
    renderComponentContent: (model) => ({
      default: (attrs: any) => {
        return (
          <TreeSelectPanel
            expand-all-on-init={true}
            treeData={attrs.treeData}
            v-model:checkStrictly={model.deptCheckStrictly}
            v-model:value={model.deptIds}
          />
        );
      },
    }),
    fieldName: 'deptIds',
    help: '更改后立即生效',
    label: '部门权限',
  },
];
