import type { BaseModel, PageQuery } from '#/api/baseModel';

import { commonExport } from '#/api/helper';
import { requestClient } from '#/api/request';

export namespace EmployeeInfoApi {
  export interface Employee extends BaseModel {
    /**
     * 板块
     */
    plate?: string;
    /**
     * 项目
     */
    project?: string;
    /**
     * 一级部门
     */
    firstLevelDepartment?: string;
    /**
     * 二级部门
     */
    secondLevelDepartment?: string;
    /**
     * 工号
     */
    employeeId?: number;

    /**
     * 职级
     */
    jobLevel?: string;
    /**
     * 岗位
     */
    position?: string;
    /**
     * 职务
     */
    post?: string;
    /**
     * 姓名
     */
    name?: string;
    /**
     * 性别
     */
    gender?: string;
    /**
     * 身份证号码
     */
    idCardNumber?: number;

    /**
     * 出生日期
     */
    birthDate?: string;
    /**
     * 年龄
     */
    age?: number;

    /**
     * 年龄段
     */
    ageGroup?: string;
    /**
     * 籍贯
     */
    nativePlace?: string;
    /**
     * 民族
     */
    ethnicity?: string;
    /**
     * 婚姻状况
     */
    maritalStatus?: string;
    /**
     * 政治面貌
     */
    politicalStatus?: string;
    /**
     * 手机号码
     */
    phoneNumber?: string;
    /**
     * 紧急联系人
     */
    emergencyContact?: string;
    /**
     * 紧急联系人电话
     */
    emergencyContactPhone?: string;
    /**
     * 家庭地址
     */
    homeAddress?: string;
    /**
     * 户口所在地
     */
    householdRegistrationAddress?: string;
    /**
     * 参加工作时间
     */
    workStartDate?: string;
    /**
     * 入职时间
     */
    entryDate?: string;
    /**
     * 工龄
     */
    yearsOfService?: number;

    /**
     * 工龄段
     */
    yearsOfServiceSegment?: string;
    /**
     * 学历
     */
    education?: string;
    /**
     * 学位
     */
    degree?: string;
    /**
     * 毕业时间
     */
    graduationDate?: string;
    /**
     * 专业
     */
    major?: string;
    /**
     * 毕业院校
     */
    graduateSchool?: string;
    /**
     * 员工类型
     */
    employeeType?: string;
    /**
     * 职称情况
     */
    professionalTitle?: string;
    /**
     * 简历
     */
    resume?: string;
    /**
     * 用工形式
     */
    employmentForm?: string;
    /**
     * 劳动合同期限
     */
    contractTerm?: string;
    /**
     * 劳动合同开始时间
     */
    contractStartDate?: string;
    /**
     * 劳动合同截止时间
     */
    contractEndDate?: string;
    /**
     * 合同到期提醒
     */
    contractExpirationReminder?: string;
    /**
     * 劳动合同签订情况
     */
    contractSigningStatus?: string;
    /**
     * 合同主体
     */
    contractEntity?: string;
    /**
     * 转正时间
     */
    regularizationDate?: string;
    /**
     * 异动情况
     */
    transferStatus?: string;
    /**
     * 奖惩情况
     */
    rewardPunishmentStatus?: string;
    /**
     * 备注
     */
    remarks?: string;
    /**
     * 离职时间
     */
    resignationDate?: string;
    /**
     * 离职原因
     */
    resignationReason?: string;
  }
}

/**
 * 查询员工信息列表
 * @param params
 * @returns {*} page
 */
export function apiPage(params: PageQuery) {
  return requestClient.get('/employee/info/page', { params });
}
/**
 * 查询员工信息详细
 * @param id
 */
export function apiDetail(id: number) {
  return requestClient.get(`/employee/info/${id}`);
}
/**
 * 新增员工信息
 * @param data
 */
export function apiAdd(data: EmployeeInfoApi.Employee) {
  return requestClient.post('/employee/info', data);
}
/**
 * 修改员工信息
 * @param data
 */
export function apiUpdate(data: EmployeeInfoApi.Employee) {
  return requestClient.put('/employee/info', data);
}
/**
 * 删除员工信息
 * @param id
 */
export function apiDelete(id: Array<number> | number) {
  return requestClient.delete(`/employee/info/${id}`);
}
/**
 * 导出员工信息
 * @param params
 */
export function apiExport(id: number) {
  return commonExport(`/employee/info/export/${id}`, {});
}

/**
 * 申请员工离职
 * @param id
 */
export function applyResign(id: number) {
  return requestClient.put(`/employee/info/applyResign/${id}`);
}

/**
 * 下载用户导入模板
 * @returns blob
 */
export function downloadImportTemplate() {
  return commonExport('/employee/info/importTemplate', {});
}

/**
 * 从excel导入用户
 * @param file
 * @returns void
 */
export function employeeImportData(file: Blob) {
  return requestClient.post<{ errMessage: string; success: boolean }>(
    '/employee/info/excel/import',
    { file },
    {
      headers: {
        'Content-Type': 'multipart/form-data;charset=UTF-8',
      },
      isTransformResponse: false,
      responseReturn: 'body',
    },
  );
}
