package com.antai.sport.http.server.server.api.business.teamtraining.service;

import com.antai.sport.http.server.common.exception.BusinessException;
import com.antai.sport.http.server.constants.DeleteStatus;
import com.antai.sport.http.server.constants.GameMode;
import com.antai.sport.http.server.constants.SportType;
import com.antai.sport.http.server.constants.TeamTrainingPlayerRecordStatus;
import com.antai.sport.http.server.repository.sport.entity.SportUser;
import com.antai.sport.http.server.repository.sport.mapper.SportUserMapper;
import com.antai.sport.http.server.repository.teamtraining.entity.*;
import com.antai.sport.http.server.repository.teamtraining.mapper.*;
import com.antai.sport.http.server.server.api.business.praise.service.PraiseService;
import com.antai.sport.http.server.server.api.business.teamtraining.converter.TeamTrainingConverter;
import com.antai.sport.http.server.server.api.business.teamtraining.dto.*;
import com.antai.sport.http.server.server.api.business.teamtraining.mapper.TeamTrainingBusinessMapper;
import com.antai.sport.http.server.server.api.business.traininglog.service.SportUserTrainingLogService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;

@Service
public class TeamTrainingService {

    @Resource
    private TeamTrainingConverter teamTrainingConverter;

    @Resource
    private TeamTrainingBannerMapper teamTrainingBannerMapper;
    @Resource
    private TeamTrainingPlayerRegistLogMapper teamTrainingPlayerRegistLogMapper;
    @Resource
    private TeamTrainingMapper teamTrainingMapper;
    @Resource
    private TeamTrainingLeaderMapper teamTrainingLeaderMapper;
    @Resource
    private TeamTrainingPlayerRecordMapper teamTrainingPlayerRecordMapper;
    @Resource
    private SportUserMapper sportUserMapper;

    @Resource
    private TeamTrainingBusinessMapper teamTrainingBusinessMapper;

    @Resource
    private PraiseService praiseService;
    @Resource
    private SportUserTrainingLogService sportUserTrainingLogService;

    /**
     * 获取日常赛banner列表
     *
     * @return
     */
    public List<TeamTrainingBannerResponseDTO> getTeamTrainingBannerList() {
        QueryWrapper<TeamTrainingBanner> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("deleted", DeleteStatus.UNDELETED).orderBy(true, true, "show_order");
        List<TeamTrainingBanner> bannerList = teamTrainingBannerMapper.selectList(queryWrapper);
        return teamTrainingConverter.toTeamTrainingBannerResponseDTO(bannerList);
    }

    /**
     * 通过用户id查看未完赛团练列表
     *
     * @param userId
     * @return
     */
    public List<TeamTrainingForListResponseDTO> getUnFinishedTeamTrainingForUser(Long userId) {
        return teamTrainingBusinessMapper.getUnFinishedTeamTrainingForUser(userId, LocalDateTime.now());
    }

    /**
     * 查询所有未完团练列表
     *
     * @return
     */
    public List<TeamTrainingForListResponseDTO> getUnFinishedTeamTrainingForServer() {
        return teamTrainingBusinessMapper.getUnFinishedTeamTrainingForServer(LocalDateTime.now());
    }

    /**
     * 团练报名
     *
     * @param param
     */
    public void regist(TeamTrainingRegistRequestDTO param) {
        QueryWrapper<TeamTrainingPlayerRegistLog> countQuery = new QueryWrapper<>();
        countQuery.eq("team_training_id", param.getTeamTrainingId()).eq("player_id", param.getPlayerId());
        Integer count = teamTrainingPlayerRegistLogMapper.selectCount(countQuery);
        if (count > 0) {
            throw new BusinessException("不成重复报名");
        }
        TeamTrainingPlayerRegistLog registLog = new TeamTrainingPlayerRegistLog();
        registLog.setTeamTrainingId(param.getTeamTrainingId());
        registLog.setPlayerId(param.getPlayerId());
        registLog.setCreateTime(LocalDateTime.now());
        teamTrainingPlayerRegistLogMapper.insert(registLog);
    }

    /**
     * 查询团练详情
     *
     * @param param
     * @return
     */
    public TeamTrainingInfoResponseDTO teamTrainingInfo(TeamTrainingInfoRequestDTO param) {
        TeamTraining teamTraining = teamTrainingMapper.selectById(param.getTeamTrainingId());
        TeamTrainingInfoResponseDTO result = teamTrainingConverter.toTeamTrainingInfoResponseDTO(teamTraining);
        QueryWrapper<TeamTrainingPlayerRegistLog> countQuery = new QueryWrapper<>();
        countQuery.eq("team_training_id", param.getTeamTrainingId());
        Integer registeredNum = teamTrainingPlayerRegistLogMapper.selectCount(countQuery);
        result.setRegisteredNum(registeredNum);
        countQuery.eq("player_id", param.getPlayerId());
        Boolean registered = teamTrainingPlayerRegistLogMapper.selectCount(countQuery) > 0;
        result.setRegistered(registered);
        TeamTrainingLeader leader = teamTrainingLeaderMapper.selectById(teamTraining.getLeaderId());
        result.setLeaderName(leader.getRealName());
        result.setAvatar(leader.getAvatar());
        return result;
    }

    /**
     * 获取团练报名信息
     *
     * @param teamTrainingId
     * @return
     */
    public List<Long> getRegisterList(Long teamTrainingId) {
        return teamTrainingBusinessMapper.getRegisterList(teamTrainingId);
    }

    /**
     * 查询团练排名
     *
     * @param teamTrainingId
     * @return
     */
    public List<TeamTrainingRankDTO> rank(Long teamTrainingId) {
        return teamTrainingBusinessMapper.getTrainingRank(teamTrainingId);
    }

    /**
     * 初始化玩家记录
     *
     * @param param
     * @return
     */
    public TeamTrainingPlayerRecordInitResponseDTO recordInit(TeamTrainingPlayerRecordInitRequestDTO param) {
        TeamTrainingPlayerRecordInitResponseDTO result = new TeamTrainingPlayerRecordInitResponseDTO();
        result.setSportType(SportType.BICYCLE);
        result.setGameMode(GameMode.GROUP_TRAINING);
        QueryWrapper<TeamTrainingPlayerRecord> countQuery = new QueryWrapper<>();
        countQuery.eq("team_training_id", param.getTeamTrainingId());
        countQuery.eq("player_id", param.getPlayerId());
        Boolean isRepeat = teamTrainingPlayerRecordMapper.selectCount(countQuery) > 0;
        if (isRepeat) {
            throw new BusinessException("您已进入不能重复进入");
        }

        TeamTrainingPlayerRecord record = new TeamTrainingPlayerRecord();
        record.setTeamTrainingId(param.getTeamTrainingId());
        record.setPlayerId(param.getPlayerId());
        record.setEntryTime(LocalDateTime.now());
        record.setStatus(TeamTrainingPlayerRecordStatus.UNFINISHED);
        record.setFinished(false);
        teamTrainingPlayerRecordMapper.insert(record);
        result.setRecordId(record.getId());
        return result;
    }


    /**
     * 比赛记录上传
     */
    @Transactional
    public void recordUpload(List<TeamTrainingRecordUploadDTO> dataList) {
        dataList.forEach(item -> {
            TeamTrainingPlayerRecord oldRecord = teamTrainingPlayerRecordMapper.selectById(item.getRecordId());
            if (!oldRecord.getFinished()) {
                SportUser user = sportUserMapper.selectById(oldRecord.getPlayerId());
                item.setWeight(user.getWeight());
                TeamTrainingPlayerRecord record = teamTrainingConverter.toTeamTrainingPlayerRecord(item, true);
                record.setPraiseNum(praiseService.getPraiseByGameModeAndRecordId(GameMode.GROUP_TRAINING, item.getRecordId()));
                teamTrainingPlayerRecordMapper.updateById(record);

                //生成骑行日志
                TeamTraining teamTraining = teamTrainingMapper.selectById(oldRecord.getTeamTrainingId());
                sportUserTrainingLogService.createBicycleTrainingLog(oldRecord.getPlayerId(), GameMode.GROUP_TRAINING,
                        teamTraining.getId(), teamTraining.getName(), teamTraining.getMap(), teamTraining.getPath(), record.getPraiseNum(), item);
            }
        });
    }
}
