package com.antai.sport.http.server.server.api.business.simplematch.service;

import com.antai.sport.http.server.common.exception.BusinessException;
import com.antai.sport.http.server.constants.DeleteStatus;
import com.antai.sport.http.server.constants.GameMode;
import com.antai.sport.http.server.constants.SportType;
import com.antai.sport.http.server.repository.simplematch.entity.SimpleMatch;
import com.antai.sport.http.server.repository.simplematch.entity.SimpleMatchBanner;
import com.antai.sport.http.server.repository.simplematch.entity.SimpleMatchPlayerRecord;
import com.antai.sport.http.server.repository.simplematch.entity.SimpleMatchPlayerRegistLog;
import com.antai.sport.http.server.repository.simplematch.mapper.SimpleMatchBannerMapper;
import com.antai.sport.http.server.repository.simplematch.mapper.SimpleMatchMapper;
import com.antai.sport.http.server.repository.simplematch.mapper.SimpleMatchPlayerRecordMapper;
import com.antai.sport.http.server.repository.simplematch.mapper.SimpleMatchPlayerRegistLogMapper;
import com.antai.sport.http.server.server.api.business.praise.service.PraiseService;
import com.antai.sport.http.server.server.api.business.simplematch.converter.SimpleMatchConverter;
import com.antai.sport.http.server.server.api.business.simplematch.dto.*;
import com.antai.sport.http.server.server.api.business.simplematch.mapper.SimpleMatchBusinessMapper;
import com.antai.sport.http.server.server.api.business.traininglog.service.SportUserTrainingLogService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;

@Service
public class SimpleMatchService {
    @Resource
    private SimpleMatchMapper simpleMatchMapper;
    @Resource
    private SimpleMatchBannerMapper simpleMatchBannerMapper;
    @Resource
    private SimpleMatchPlayerRegistLogMapper simpleMatchPlayerRegistLogMapper;
    @Resource
    private SimpleMatchPlayerRecordMapper simpleMatchPlayerRecordMapper;

    @Resource
    private SimpleMatchBusinessMapper simpleMatchBusinessMapper;

    @Resource
    private SimpleMatchConverter simpleMatchConverter;

    @Resource
    private PraiseService praiseService;
    @Resource
    private SportUserTrainingLogService sportUserTrainingLogService;

    /**
     * 创建比赛
     *
     * @param initData
     */
    public void init(ReqSimpleMatchInit initData) {
        if (StringUtils.isBlank(initData.getName())) {
            throw new BusinessException("比赛名称不能为空");
        }

        if (initData.getEntryTime() == null || initData.getStartTime() == null ||
                initData.getEndTime() == null) {
            throw new BusinessException("所有输入时间不能为空");
        }

        if (initData.getEntryTime().equals(initData.getStartTime()) ||
                initData.getEntryTime().isAfter(initData.getStartTime())) {
            throw new BusinessException("允许进入时间必须早于比赛开始时间");
        }

        if (initData.getStartTime().equals(initData.getEndTime()) ||
                initData.getStartTime().isAfter(initData.getEndTime())) {
            throw new BusinessException("比赛开始时间必须早于比赛结束时间");
        }

        SimpleMatch simpleMatch = simpleMatchConverter.toSimpleMatch(initData);
        simpleMatchMapper.insert(simpleMatch);
    }

    /**
     * 获取日常赛banner列表
     *
     * @return
     */
    public List<RespSimpleMatchBanner> getSimpleMatchBannerList() {
        QueryWrapper<SimpleMatchBanner> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("deleted", DeleteStatus.UNDELETED)
                .orderBy(true, true, "show_order");
        List<SimpleMatchBanner> bannerList = simpleMatchBannerMapper.selectList(queryWrapper);
        return simpleMatchConverter.toRespBannerList(bannerList);
    }

    /**
     * 通过用户id查看未完赛事列表
     *
     * @param userId
     * @return
     */
    public List<RespSimpleMatchList> getUnFinishedSimpleMatchForUser(Long userId) {
        return simpleMatchBusinessMapper.getUnFinishedSimpleMatchListForUser(userId, LocalDateTime.now());
    }

    /**
     * 查询所有未完赛事列表
     *
     * @return
     */
    public List<RespSimpleMatchList> getUnFinishedSimpleMatchForServer() {
        return simpleMatchBusinessMapper.getUnFinishedSimpleMatchListForServer(LocalDateTime.now());
    }

    /**
     * 比赛报名
     *
     * @param param
     */
    public void regist(ReqSimpleMatchRegist param) {
        QueryWrapper<SimpleMatchPlayerRegistLog> countQuery = new QueryWrapper<>();
        countQuery.eq("match_id", param.getMatchId()).eq("player_id", param.getPlayerId());
        Integer count = simpleMatchPlayerRegistLogMapper.selectCount(countQuery);
        if (count > 0) {
            throw new BusinessException("不成重复报名");
        }
        SimpleMatchPlayerRegistLog registLog = new SimpleMatchPlayerRegistLog();
        registLog.setMatchId(param.getMatchId());
        registLog.setPlayerId(param.getPlayerId());
        registLog.setCreateTime(LocalDateTime.now());
        simpleMatchPlayerRegistLogMapper.insert(registLog);
    }

    /**
     * 查询赛事详情
     *
     * @param param
     * @return
     */
    public RespSimpleMatchInfo matchInfo(ReqMatchInfo param) {
        SimpleMatch match = simpleMatchMapper.selectById(param.getMatchId());
        RespSimpleMatchInfo matchResp = simpleMatchConverter.toRespSimpleMatchInfo(match);
        QueryWrapper<SimpleMatchPlayerRegistLog> countQuery = new QueryWrapper<>();
        countQuery.eq("match_id", param.getMatchId());
        Integer registeredNum = simpleMatchPlayerRegistLogMapper.selectCount(countQuery);
        matchResp.setRegisteredNum(registeredNum);
        countQuery.eq("player_id", param.getPlayerId());
        Boolean registered = simpleMatchPlayerRegistLogMapper.selectCount(countQuery) > 0;
        matchResp.setRegistered(registered);
        return matchResp;
    }

    /**
     * 初始化玩家进入近路
     *
     * @param param
     * @return
     */
    public RespSimpleMatchRecordInit recordInit(ReqSimpleMatchRecordInit param) {
        RespSimpleMatchRecordInit resp = new RespSimpleMatchRecordInit();
        resp.setSportType(SportType.BICYCLE);
        resp.setGameMode(GameMode.SIMPLE_MATCH);
        QueryWrapper<SimpleMatchPlayerRecord> countQuery = new QueryWrapper<>();
        countQuery.eq("match_id", param.getMatchId());
        countQuery.eq("player_id", param.getPlayerId());
        Boolean isRepeat = simpleMatchPlayerRecordMapper.selectCount(countQuery) > 0;
        if (isRepeat) {
            throw new BusinessException("您已进入不能重复进入");
        }

        SimpleMatchPlayerRecord record = new SimpleMatchPlayerRecord();
        record.setMatchId(param.getMatchId());
        record.setPlayerId(param.getPlayerId());
        record.setEntryTime(LocalDateTime.now());
        simpleMatchPlayerRecordMapper.insert(record);
        resp.setRecordId(record.getId());
        return resp;
    }

    /**
     * 比赛记录上传
     */
    @Transactional
    public void recordUpload(List<ReqSimpleMatchRecord> dataList) {
        dataList.forEach(item -> {
            SimpleMatchPlayerRecord oldRecord = simpleMatchPlayerRecordMapper.selectById(item.getRecordId());
            if (oldRecord.getStatus() == null) {
                SimpleMatchPlayerRecord record = simpleMatchConverter.toSimpleMatchPlayerRecord(item);
                record.setPraiseNum(praiseService.getPraiseByGameModeAndRecordId(GameMode.SIMPLE_MATCH, item.getRecordId()));
                simpleMatchPlayerRecordMapper.updateById(record);

                //生成骑行日志
                SimpleMatch match = simpleMatchMapper.selectById(oldRecord.getMatchId());
                sportUserTrainingLogService.createBicycleTrainingLog(oldRecord.getPlayerId(), GameMode.SIMPLE_MATCH,
                        oldRecord.getId(), match.getMap(), match.getPath(), record.getPraiseNum(), item);
            }
        });
    }

    /**
     * 查询日常赛排名
     *
     * @param matchId
     * @return
     */
    public List<RespSimpleMatchRank> rank(Long matchId) {
        return simpleMatchBusinessMapper.getSimpleMatchRank(matchId);
    }
}
