package com.antai.sport.http.server.common.util;

import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.dysmsapi.model.v20170525.SendSmsRequest;
import com.aliyuncs.dysmsapi.model.v20170525.SendSmsResponse;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Map;

/**
 * @author 厉明
 * @date 2019/12/17 10:14
 * @email lmmax@126.com
 * @description 短信发送工具
 */
@Service
@Slf4j
public class SmsUtil {
    @Value("${aliyun.access-key-id}")
    private String accessKeyId;
    @Value("${aliyun.access-key-secret}")
    private String accessKeySecret;
    @Value("${aliyun.sms.region-id}")
    private String regionId;
    @Value("${aliyun.sms.time-out}")
    private String timeOut;
    static final String PRODUCT = "Dysmsapi";
    static final String DOMAIN = "dysmsapi.aliyuncs.com";
    static final String SUCCEED_STATUS = "OK";
    @Resource
    private ObjectMapper objectMapper;

    /**
     * 发送短信
     *
     * @param sign          签名
     * @param templateCode  短信模板
     * @param templateParam 模板参数
     * @param phoneNums     电话号码
     * @return 短信发送结果
     * @throws ClientException         客户端异常
     * @throws JsonProcessingException json转换异常
     */
    public String sendSms(String sign, String templateCode, Map<String, Object> templateParam, String... phoneNums) throws ClientException, JsonProcessingException {
        if (phoneNums.length > 1000) {
            return "最大发送数量不能超过1000";
        }
        //设置超时时间-可自行调整
        System.setProperty("sun.net.client.defaultConnectTimeout", timeOut);
        System.setProperty("sun.net.client.defaultReadTimeout", timeOut);
        //初始化ascClient需要的几个参数
        //初始化ascClient,暂时不支持多region（请勿修改）
        IClientProfile profile = DefaultProfile.getProfile(regionId, accessKeyId, accessKeySecret);
        DefaultProfile.addEndpoint(regionId, PRODUCT, DOMAIN);
        IAcsClient acsClient = new DefaultAcsClient(profile);
        //组装请求对象
        SendSmsRequest request = new SendSmsRequest();
        //必填:待发送手机号。支持以逗号分隔的形式进行批量调用，批量上限为1000个手机号码,批量调用相对于单条调用及时性稍有延迟,验证码类型的短信推荐使用单条调用的方式；发送国际/港澳台消息时，接收号码格式为00+国际区号+号码，如“0085200000000”
        String phoneNumbsers = StringUtils.join(phoneNums, ",");
        request.setPhoneNumbers(phoneNumbsers);
        //必填:短信签名-可在短信控制台中找到
        request.setSignName(sign);
        //必填:短信模板-可在短信控制台中找到
        request.setTemplateCode(templateCode);
        //可选:模板中的变量替换JSON串,如模板内容为"亲爱的${name},您的验证码为${code}"时,此处的值为
        //友情提示:如果JSON中需要带换行符,请参照标准的JSON协议对换行符的要求,比如短信内容中包含\r\n的情况在JSON中需要表示成\\r\\n,否则会导致JSON在服务端解析失败
        //阿里云单个变量长度限制在20字符内
        request.setTemplateParam(objectMapper.writeValueAsString(templateParam));
        //请求失败这里会抛ClientException异常
        SendSmsResponse sendSmsResponse = acsClient.getAcsResponse(request);
        if (sendSmsResponse.getCode() != null && SUCCEED_STATUS.equals(sendSmsResponse.getCode())) {
            log.info("阿里云短信发送成功:" + StringUtils.join(phoneNums));
            return null;
        } else {
            log.error("阿里云短信发送失败状态码:" + StringUtils.join(phoneNums) + "-" + sendSmsResponse.getCode());
            return sendSmsResponse.getCode();
        }
    }
}
