package com.antai.sport.http.server.mobile.api.business.sport.user;


import com.antai.sport.http.server.common.base.Result;
import com.antai.sport.http.server.common.exception.BusinessException;
import com.antai.sport.http.server.common.jwt.TokenService;
import com.antai.sport.http.server.common.jwt.TokenVO;
import com.antai.sport.http.server.common.service.CommonCodeService;
import com.antai.sport.http.server.common.service.CommonSmsService;
import com.antai.sport.http.server.common.util.RandomUtil;
import com.antai.sport.http.server.common.vo.SysCodeDetailVO;
import com.antai.sport.http.server.constants.DeleteStatus;
import com.antai.sport.http.server.constants.LoginMode;
import com.antai.sport.http.server.constants.SportType;
import com.antai.sport.http.server.constants.SysCodeConstants;
import com.antai.sport.http.server.mobile.api.business.sport.user.dto.*;
import com.antai.sport.http.server.repository.base.mapper.BaseAvatarMapper;
import com.antai.sport.http.server.repository.sport.entity.*;
import com.antai.sport.http.server.repository.sport.mapper.*;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.DigestUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * @Author liming
 * @Date 2021/8/21 19:41
 * @Email lmmax@126.com
 * @Description:
 */
@Service
public class SportUserService {
    @Value("${spring.cache.redis.keys.login-sms-captcha}")
    private String loginSmsCaptchaRedisKey;
    @Value("${spring.cache.redis.expire.login-sms-captcha}")
    private long loginSmsCaptchaExpire;
    @Value("#{'${spring.profiles.active}'.equals('prod')}")
    private boolean isProd;
    @Resource
    private SportUserSummaryMapper sportUserSummaryMapper;
    @Resource
    private SportUserMapper sportUserMapper;
    @Resource
    private SportSectionMapper sportSectionMapper;
    @Resource
    private SportSectionSkinMapper sportSectionSkinMapper;
    @Resource
    private SportUserSectionSkinMapper sportUserSectionSkinMapper;
    @Resource
    private BaseAvatarMapper baseAvatarMapper;
    @Resource
    private SportUserSectionSkinService sportUserSectionSkinService;
    @Resource
    private TokenService tokenService;
    @Resource
    private CommonSmsService commonSmsService;
    @Resource
    private CommonCodeService commonCodeService;
    @Resource
    private RedisTemplate<String, Object> redisTemplate;


    public Result login(ReqLogin param) {
        if (param.getMode() == null || (!param.getMode().equals(LoginMode.PASSWORD) && !param.getMode().equals(LoginMode.SMS_CAPTCHA))) {
            throw new BusinessException("请选择正确的登录模式");
        }
        if (StringUtils.isBlank(param.getUsername())) {
            throw new BusinessException("请输入用户名");
        }
        List<SportUser> sportUserList = sportUserMapper.selectList(new QueryWrapper<SportUser>()
                .eq("phone", param.getUsername())
                .eq("deleted", DeleteStatus.UNDELETED));


        Result result = new Result();
        SportUser sportUser;
        if (param.getMode().equals(LoginMode.PASSWORD)) {
            if (sportUserList.isEmpty()) {
                throw new BusinessException("用户不存在，请先使用手机短信验证码登录");
            }
            sportUser = sportUserList.get(0);
            if (sportUser.getPassword() == null) {
                throw new BusinessException("当前用户还没有设置密码，请使用验证码方式登录，在个人设置中设置密码");
            }
            if (StringUtils.isBlank(param.getPassword())) {
                throw new BusinessException("请输入密码");
            }
            String hashPwd = DigestUtils.md5DigestAsHex(param.getPassword().getBytes());
            if (!hashPwd.equals(sportUser.getPassword())) {
                throw new BusinessException("用户名或密码错误");
            }
        } else {
            if (StringUtils.isBlank(param.getSmsCaptcha())) {
                throw new BusinessException("请输入验证码");
            }
            String redisKey = loginSmsCaptchaRedisKey + param.getUsername();
            Object redisValue = redisTemplate.opsForValue().get(redisKey);
            if (redisValue == null) {
                throw new BusinessException("请先获取验证码");
            }
            String loginSmsCaptcha = redisValue.toString();
            if (!loginSmsCaptcha.equals(param.getSmsCaptcha())) {
                throw new BusinessException("短信验证码错误");
            }
            if (sportUserList.isEmpty()) {
                sportUser = new SportUser();
                sportUser.setPhone(param.getUsername());
                sportUser.setCtDate(LocalDateTime.now());
                sportUser.setRoleInitialized(false);
                sportUser.setInfoInitialized(false);
                sportUser.setEquipmentInitialized(false);
                sportUserMapper.insert(sportUser);
            } else {
                sportUser = sportUserList.get(0);
            }
            redisTemplate.delete(redisKey);
        }
        result.setRespMsg("登录成功");
        TokenVO tokenVO = tokenService.initToken(param.getUsername());
//        RespLogin respLogin = new RespLogin();
//        respLogin.setInfoInitialized(sportUser.getInfoInitialized());
//        respLogin.setRoleInitialized(sportUser.getRoleInitialized());
//        respLogin.setEquipmentInitialized(sportUser.getEquipmentInitialized());
//        result.setData(respLogin);
        result.setLongToken(tokenVO.getLongToken());
        result.setShortToken(tokenVO.getShortToken());
        result.setData(sportUser.getId());
        return result;
    }

    /**
     * 发送验证码逻辑
     *
     * @param phone 电话号码
     */
    public String sendLoginSmsCaptcha(String phone) {
        String code = String.valueOf(RandomUtil.generateNum());
        redisTemplate.opsForValue().set(loginSmsCaptchaRedisKey + phone, code, loginSmsCaptchaExpire, TimeUnit.SECONDS);
        return code;
//        if (isProd) {
//            commonSmsService.sendLoginSmsCaptcha(phone, code);
//            return null;
//        } else {
//            return code;
//        }
    }

    /**
     * 获取注册页面基础数据
     *
     * @return
     */
    public RespRegisterInfo getRegisterInfo() {
        RespRegisterInfo respRegisterInfo = new RespRegisterInfo();
        Map<String, List<SysCodeDetailVO>> codeMap = commonCodeService.getCodeByTypes(SysCodeConstants.SEX);
        respRegisterInfo.setSexCode(codeMap.get(SysCodeConstants.SEX));
        respRegisterInfo.setAvatarList(baseAvatarMapper.selectList(null));
        return respRegisterInfo;
    }

    /**
     * 保存用户基本信息
     *
     * @param loginUser
     * @param param
     */
    public void saveUserInfo(SportUser loginUser, ReqSaveUserInfo param) {
        if (param.getNickname() != null) {
            if (sportUserMapper.checkNicknameRepeat(loginUser.getId(), param.getNickname())) {
                throw new BusinessException("当前昵称已被使用!");
            }
        }
        loginUser.setInfoInitialized(true);
        loginUser.setNickname(param.getNickname());
        loginUser.setSex(param.getSex());
        loginUser.setBirthday(param.getBirthday().atStartOfDay());
        loginUser.setHeight(param.getHeight());
        loginUser.setWeight(param.getWeight());
        loginUser.setAvatar(param.getAvatar());
        loginUser.setFtp(param.getFtp());
        loginUser.setProvince(param.getProvince());
        loginUser.setCity(param.getCity());
        loginUser.setProvinceName(param.getProvinceName());
        loginUser.setCityName(param.getCityName());
        loginUser.setHeartRate(param.getHeartRate());
        sportUserMapper.updateById(loginUser);
        redisTemplate.delete("sport_user::" + loginUser.getPhone());
    }

    /**
     * @param loginUser
     * @return
     */
    public List<RespSportSection> getUserSection(Integer sectionType, SportUser loginUser) {
        if (loginUser.getSex() == null) {
            throw new BusinessException("请先选择姓名别");
        }
        List<RespSportSection> resultList = new ArrayList<>();

        QueryWrapper<SportSection> sectionQueryWrapper = new QueryWrapper<>();
        sectionQueryWrapper.eq("type", sectionType);
        sectionQueryWrapper.orderBy(true, true, Arrays.asList("show_order", "id"));
        List<SportSection> sectionList = sportSectionMapper.selectList(sectionQueryWrapper);

        QueryWrapper<SportSectionSkin> sectionSkinQueryWrapper = new QueryWrapper<>();
        sectionSkinQueryWrapper.eq("type", sectionType);
        sectionSkinQueryWrapper.eq("sex", loginUser.getSex());
        sectionSkinQueryWrapper.orderBy(true, true, Arrays.asList("show_order", "id"));
        List<SportSectionSkin> sectionSkinList = sportSectionSkinMapper.selectList(sectionSkinQueryWrapper);

        Map<Long, String> skinKeyMap = new HashMap<>();
        Map<Long, List<RespSportSectionSkin>> sectionSkinMap = new HashMap<>();
        for (SportSectionSkin item : sectionSkinList) {
            skinKeyMap.put(item.getId(), item.getSkinKey());
            if (!sectionSkinMap.containsKey(item.getSection())) {
                sectionSkinMap.put(item.getSection(), new ArrayList<>());
            }
            sectionSkinMap.get(item.getSection()).add(new RespSportSectionSkin(item));
        }

        Map<Long, String> userSkinMap = new HashMap<>();
        QueryWrapper<SportUserSectionSkin> userSkinQueryWrapper = new QueryWrapper();
        userSkinQueryWrapper.eq("sport_user_id", loginUser.getId());
        userSkinQueryWrapper.eq("type", sectionType);
        List<SportUserSectionSkin> userSkinList = sportUserSectionSkinMapper.selectList(userSkinQueryWrapper);
        for (SportUserSectionSkin item : userSkinList) {
            if (skinKeyMap.containsKey(item.getSkin())) {
                userSkinMap.put(item.getSection(), skinKeyMap.get(item.getSkin()));
            }
        }

        for (SportSection item : sectionList) {
            if (sectionSkinMap.containsKey(item.getId())) {
                RespSportSection section = new RespSportSection(item);
                section.setSkinList(sectionSkinMap.get(item.getId()));
                if (userSkinMap.containsKey(item.getId())) {
                    section.setSkinKey(userSkinMap.get(item.getId()));
                } else {
                    section.setSkinKey(section.getSkinList().get(0).getKey());
                }
                resultList.add(section);
            }

        }

        return resultList;
    }

    /**
     * 保存用户皮肤
     *
     * @param sectionType
     * @param loginUser
     * @param skinList
     */
    @Transactional
    public void saveUserSkin(Integer sectionType, SportUser loginUser, List<ReqSaveUserSkin> skinList) {
        List<Long> delUserSectionList = new ArrayList<>(skinList.size());
        List<SportUserSectionSkin> saveList = new ArrayList<>();
        for (ReqSaveUserSkin item : skinList) {
            delUserSectionList.add(item.getSection());
            SportUserSectionSkin userSkin = new SportUserSectionSkin();
            userSkin.setSportUserId(loginUser.getId());
            userSkin.setSection(item.getSection());
            userSkin.setSkin(item.getSkin());
            userSkin.setType(sectionType);
            saveList.add(userSkin);
        }
        QueryWrapper<SportUserSectionSkin> userSkinQueryWrapper = new QueryWrapper<>();
        userSkinQueryWrapper.eq("sport_user_id", loginUser.getId());
        userSkinQueryWrapper.in("section", delUserSectionList);
        sportUserSectionSkinMapper.delete(userSkinQueryWrapper);
        sportUserSectionSkinService.saveBatch(saveList);

    }

    @Transactional
    public void saveUserSkin(SportUser loginUser, List<ReqSaveUserSkin> skinList) {
        List<String> delUserSectionList = new ArrayList<>(skinList.size());
        List<SportUserSectionSkin> saveList = new ArrayList<>();
        for (ReqSaveUserSkin item : skinList) {
            delUserSectionList.add(item.getSectionKey());
            SportUserSectionSkin userSkin = new SportUserSectionSkin();
            userSkin.setSportUserId(loginUser.getId());
            userSkin.setSectionKey(item.getSectionKey());
            userSkin.setSkinKey(item.getSkinKey());
            saveList.add(userSkin);
        }
        QueryWrapper<SportUserSectionSkin> userSkinQueryWrapper = new QueryWrapper<>();
        userSkinQueryWrapper.eq("sport_user_id", loginUser.getId());
        userSkinQueryWrapper.in("section_key", delUserSectionList);
        sportUserSectionSkinMapper.delete(userSkinQueryWrapper);
        sportUserSectionSkinService.saveBatch(saveList);
    }

    public Map<String, String> getUserSkin(SportUser loginUser) {
        Map<String, String> result = null;
        QueryWrapper<SportUserSectionSkin> userSkinQueryWrapper = new QueryWrapper<>();
        userSkinQueryWrapper.eq("sport_user_id", loginUser.getId());
        List<SportUserSectionSkin> skinList = sportUserSectionSkinMapper.selectList(userSkinQueryWrapper);
        if (skinList.size() > 0) {
            result = new HashMap<>();
            for (SportUserSectionSkin item : skinList) {
                result.put(item.getSectionKey(), item.getSkinKey());
            }
        }
        return result;
    }

    public RespBaseUserInfo getUserInfo(SportUser loginUser) {
        RespBaseUserInfo userInfo = new RespBaseUserInfo(loginUser);
        QueryWrapper<SportUserSummary> summaryQuery = new QueryWrapper<>();
        summaryQuery.eq("deleted", DeleteStatus.UNDELETED);
        summaryQuery.eq("user_id", loginUser.getId());
        List<SportUserSummary> summaryList = sportUserSummaryMapper.selectList(summaryQuery);
        for (SportUserSummary item : summaryList) {
            if (item.getType().equals(SportType.BICYCLE)) {
                userInfo.setBicycleSummary(new RespSportUserSummary(item));
            } else {
                userInfo.setRunSummary(new RespSportUserSummary(item));
            }
        }
        if (userInfo.getBicycleSummary() == null) {
            userInfo.setBicycleSummary(new RespSportUserSummary(null));
        }
        if (userInfo.getRunSummary() == null) {
            userInfo.setRunSummary(new RespSportUserSummary(null));
        }
        return userInfo;
    }

    public List<RespUserInfo> getUserInfoByIds(List<Long> ids) {
        QueryWrapper<SportUser> queryWrapper = new QueryWrapper<>();
        queryWrapper.in("id", ids);
        return getUserInfo(queryWrapper);
    }

    public List<RespUserInfo> getUserInfoLike(String key) {
        QueryWrapper<SportUser> queryWrapper = new QueryWrapper<>();
        queryWrapper.like("nickname", key).or().like("phone", key);
        return getUserInfo(queryWrapper);
    }

    public List<RespUserInfo> getUserInfo(QueryWrapper<SportUser> queryWrapper) {
        List<SportUser> userList = sportUserMapper.selectList(queryWrapper);
        List<RespUserInfo> resultList = new ArrayList<>(userList.size());
        userList.forEach(user -> {
            RespUserInfo item = new RespUserInfo();
            item.setUserId(user.getId());
            item.setName(user.getNickname());
            item.setAvatar(user.getAvatar());
            item.setFtp(user.getFtp());
            item.setWeight(user.getWeight());
            item.setWkg(handleWkg(user.getFtp(), user.getWeight()));
            item.setProvinceName(user.getProvinceName());
            item.setCityName(user.getCityName());
            item.setHeartRate(user.getHeartRate());
            resultList.add(item);
        });
        return resultList;
    }

    private BigDecimal handleWkg(Integer sourceFtp, BigDecimal weight) {
        if (sourceFtp == null || weight == null) {
            return BigDecimal.ZERO;
        }
        BigDecimal ftp = BigDecimal.valueOf(sourceFtp);
        if (ftp.compareTo(BigDecimal.ZERO) == 0 || weight.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        } else {
            return ftp.divide(weight, 1, RoundingMode.HALF_UP);
        }
    }

}
