package com.antai.sport.http.server.server.api.business.series.service;

import com.antai.sport.http.server.common.exception.BusinessException;
import com.antai.sport.http.server.constants.GameMode;
import com.antai.sport.http.server.constants.SeriesMatchPlayerRecordStatus;
import com.antai.sport.http.server.constants.SeriesMatchType;
import com.antai.sport.http.server.constants.SportType;
import com.antai.sport.http.server.repository.series.entity.*;
import com.antai.sport.http.server.repository.series.mapper.*;
import com.antai.sport.http.server.repository.sport.entity.SportUser;
import com.antai.sport.http.server.repository.sport.mapper.SportUserMapper;
import com.antai.sport.http.server.server.api.business.base.mapper.BaseBusinessMapper;
import com.antai.sport.http.server.server.api.business.praise.service.PraiseService;
import com.antai.sport.http.server.server.api.business.series.converter.SeriesConverter;
import com.antai.sport.http.server.server.api.business.series.dto.SeriesApplyDTO;
import com.antai.sport.http.server.server.api.business.series.dto.SeriesQualifyingMatchRankDTO;
import com.antai.sport.http.server.server.api.business.series.dto.SeriesQualifyingMatchRecordInitDTO;
import com.antai.sport.http.server.server.api.business.series.mapper.SeriesBusinessMapper;
import com.antai.sport.http.server.server.api.business.series.vo.*;
import com.antai.sport.http.server.server.api.business.traininglog.service.SportUserTrainingLogService;
import com.antai.sport.http.server.server.api.common.dto.CyclingDataDTO;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.metadata.OrderItem;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

@Service
public class SeriesService {

    @Resource
    private SeriesMapper seriesMapper;
    @Resource
    private SeriesApplyMapper seriesApplyMapper;
    @Resource
    private SeriesRoundMapper seriesRoundMapper;
    @Resource
    private SportUserMapper sportUserMapper;
    @Resource
    private SeriesRoundRankMapper seriesRoundRankMapper;
    @Resource
    private SeriesMatchPlayerRecordMapper seriesMatchPlayerRecordMapper;
    @Resource
    private SeriesMatchMapper seriesMatchMapper;

    @Resource
    private SeriesConverter seriesConverter;

    @Resource
    private SeriesBusinessMapper seriesBusinessMapper;
    @Resource
    private BaseBusinessMapper baseBusinessMapper;

    @Resource
    private PraiseService praiseService;
    @Resource
    private SportUserTrainingLogService sportUserTrainingLogService;

    public List<SeriesListVO> getSeriesList() {
        return seriesBusinessMapper.getSeriesList();
    }

    public SeriesDetailsVO getSeriesDetails(Long id, Long sportUserId) {
        Series series = seriesMapper.selectById(id);
        if (series == null) {
            throw new BusinessException("您查询的系列赛信息不存在");
        }
        LocalDateTime now = LocalDateTime.now();
        Integer status;
        if (series.getFinished()) {
            status = 40;
        } else if (now.isBefore(series.getApplyStartTime())) {
            status = 30;
        } else if (now.isAfter(series.getApplyStartTime()) && now.isBefore(series.getApplyEndTime())) {
            status = 10;
        } else {
            status = 20;
        }
        SeriesDetailsVO result = seriesConverter.toSeriesDetailVO(series, status);
        SeriesApply apply = seriesBusinessMapper.getSeriesApply(id, sportUserId);
        result.setAreaId(apply.getAreaId());
        result.setNoticeList(seriesBusinessMapper.getSeriesNotice(id));
        result.setApplied(seriesBusinessMapper.checkExistUserSeriesApply(id, sportUserId));
        result.setRoundList(seriesBusinessMapper.getSeriesRound(id));
        result.setAreaList(seriesBusinessMapper.getSeriesAreaList(id));
        return result;
    }

    public SeriesApplyPreLoadVO getSeresApplyPreLoadInfo(Long sportUserId, Long seriesId) {
        SportUser sportUser = sportUserMapper.selectById(sportUserId);
        SeriesApplyPreLoadVO result = seriesConverter.toSeriesApplyPreLoadVO(sportUser);
        result.setAreaList(seriesBusinessMapper.getSeriesAreaList(seriesId));
        SeriesApply seriesApply = seriesBusinessMapper.getLatestUserSeriesApplyInfo(sportUserId);
        if (seriesApply != null) {
            result.setRealName(seriesApply.getName());
            result.setIdCard(seriesApply.getIdCard());
        }

        return result;
    }

    public void seriesApply(SeriesApplyDTO dto) {
        if (seriesBusinessMapper.checkExistUserSeriesApply(dto.getSeriesId(), dto.getSportUserId())) {
            throw new BusinessException("您已经报名当前比赛，无需重复报名");
        }

        SeriesApply seriesApply = seriesConverter.toSeriesApply(dto);
        seriesApply.setCreateTime(LocalDateTime.now());
        seriesApplyMapper.insert(seriesApply);
    }

    public SeriesQualifyingMatchHomeVO getSeriesQualifyingMatchHomeData(Long roundId, Long areaId, Long sportUserId) {
        SeriesQualifyingMatchHomeVO result = new SeriesQualifyingMatchHomeVO();
        SeriesRound seriesRound = seriesRoundMapper.selectById(roundId);
        if (seriesRound == null) {
            throw new BusinessException("当前轮次比赛不存在，请后退，重新打开当亲你赛事");
        }
        result.setChance(0);
        result.setLeftChance(0);
        result.setPath(seriesRound.getPath());
        result.setMap(seriesRound.getMap());
        result.setDistance(seriesRound.getDistance());
        result.setClimbDistance(seriesRound.getClimbDistance());
        result.setResultConfirm(seriesRound.getResultConfirm());
        result.setPathName(baseBusinessMapper.getMapPathName(seriesRound.getPath()));

        LocalDate today = LocalDate.now();
        LocalDate tomorrow = today.plusDays(1);
        result.setMatchList(seriesBusinessMapper
                .getSeriesQualifyingMatchList(roundId, areaId, sportUserId, today, tomorrow));

        SeriesApply apply = seriesBusinessMapper.getSeriesApply(seriesRound.getSeriesId(), sportUserId);
        if (apply != null) {
            result.setAreaId(apply.getAreaId());
            if (areaId.equals(apply.getAreaId())) {
                result.setChance(seriesRound.getChance());
                result.setLeftChance(seriesRound.getChance() - seriesBusinessMapper.getSeriesQualifyingMatchLeftChance(roundId, sportUserId));
                result.setBestResult(seriesBusinessMapper.getBestResult(roundId, sportUserId));

                //查询我的排名
                QueryWrapper<SeriesRoundRank> rankQuery = new QueryWrapper<>();
                rankQuery.lambda().eq(SeriesRoundRank::getRoundId, roundId)
                        .eq(SeriesRoundRank::getAreaId, areaId)
                        .eq(SeriesRoundRank::getSportUserId, sportUserId);

                SeriesRoundRank myRankData = seriesRoundRankMapper.selectOne(rankQuery);
                if (myRankData != null) {
                    Integer myRank = seriesBusinessMapper.getMySeriesQualifyingMatchRank(roundId, areaId,
                            myRankData.getDuration(), myRankData.getCreateTime());
                    result.setMyRank(myRank);
                }
            }
        }
        return result;
    }

    public List<SeriesQualifyingMatchRankVO> getSeriesQualifyingMatchRank(SeriesQualifyingMatchRankDTO dto) {
        Page<SeriesQualifyingMatchRankVO> pageParam = new Page<>(dto.getPageNo(), dto.getPageSize());
        pageParam.addOrder(new OrderItem("t1.duration", true));
        pageParam.addOrder(new OrderItem("t1.create_time", true));
        pageParam.addOrder(new OrderItem("t1.id", true));
        IPage<SeriesQualifyingMatchRankVO> result = seriesBusinessMapper.getSeriesQualifyingMatchRank(pageParam, dto);
        return result.getRecords();
    }

    public SeriesMatchRecordInitVO seriesQualifyingMatchRecordInit(SeriesQualifyingMatchRecordInitDTO dto) {
        SeriesMatch seriesMatch = seriesMatchMapper.selectById(dto.getMatchId());
        SeriesApply seriesApply = seriesBusinessMapper.getSeriesApply(seriesMatch.getSeriesId(), dto.getSportUserId());
        if (!seriesApply.getAreaId().equals(seriesMatch.getAreaId())) {
            throw new BusinessException("您没有报名当前赛区比赛，不能进入比赛!");
        }

        SeriesRound round = seriesRoundMapper.selectById(seriesMatch.getRoundId());
        Integer leftChance = seriesBusinessMapper.getSeriesQualifyingMatchLeftChance(seriesMatch.getRoundId(),
                dto.getSportUserId());
        if (leftChance.compareTo(round.getChance()) >= 0) {
            throw new BusinessException("您的预选赛参与机会已经用完！");
        }

        SeriesMatchRecordInitVO result = new SeriesMatchRecordInitVO();
        result.setSportType(SportType.BICYCLE);
        result.setGameMode(GameMode.SERIES_MATCH);
        QueryWrapper<SeriesMatchPlayerRecord> countQuery = new QueryWrapper<>();
        countQuery.lambda().eq(SeriesMatchPlayerRecord::getMatchId, dto.getMatchId())
                .eq(SeriesMatchPlayerRecord::getPlayerId, dto.getSportUserId());
        Boolean isRepeat = seriesMatchPlayerRecordMapper.selectCount(countQuery) > 0;
        if (isRepeat) {
            throw new BusinessException("您已初始化比赛数据，不能重复初始化");
        }
        SeriesMatchPlayerRecord record = new SeriesMatchPlayerRecord();
        record.setSeriesId(seriesMatch.getSeriesId());
        record.setRoundId(seriesMatch.getRoundId());
        record.setMatchId(dto.getMatchId());
        record.setPlayerId(dto.getSportUserId());
        record.setEntryTime(LocalDateTime.now());
        record.setStatus(SeriesMatchPlayerRecordStatus.UNFINISHED);
        record.setFinished(false);
        seriesMatchPlayerRecordMapper.insert(record);
        result.setRecordId(record.getId());

        return result;
    }

    /**
     * 系列赛比赛结果数据上传
     */
    @Transactional
    public void recordUpload(List<CyclingDataDTO> dataList) {
        dataList.forEach(item -> {
            SeriesMatchPlayerRecord oldRecord = seriesMatchPlayerRecordMapper.selectById(item.getRecordId());
            if (!oldRecord.getFinished()) {
                SportUser user = sportUserMapper.selectById(oldRecord.getPlayerId());
                item.setWeight(user.getWeight());
                SeriesMatchPlayerRecord record = seriesConverter.toSeriesMatchPlayerRecord(item, true);
                record.setPraiseNum(praiseService.getPraiseByGameModeAndRecordId(GameMode.SERIES_MATCH,
                        item.getRecordId()));
                seriesMatchPlayerRecordMapper.updateById(record);

                //生成骑行日志
                SeriesMatch match = seriesMatchMapper.selectById(oldRecord.getMatchId());
                Series series = seriesMapper.selectById(oldRecord.getSeriesId());
                String matchType = match.getType().equals(SeriesMatchType.QUALIFYING) ? "预选赛" : "淘汰赛";
                String sourceName = series.getName() + "(" + matchType + ":" + match.getName() + ")";

                //处理系列赛排名
                if (item.getStatus().equals(SeriesMatchPlayerRecordStatus.FINISHED)) {
                    QueryWrapper<SeriesRoundRank> rankQuery = new QueryWrapper<>();
                    rankQuery.lambda().eq(SeriesRoundRank::getRoundId, match.getRoundId())
                            .eq(SeriesRoundRank::getSportUserId, oldRecord.getPlayerId());
                    SeriesRoundRank rank = seriesRoundRankMapper.selectOne(rankQuery);

                    if (rank == null || rank.getDuration().compareTo(item.getDuration()) == 1) {
                        if (rank == null) {
                            rank = new SeriesRoundRank();
                            rank.setSeriesId(match.getSeriesId());
                            rank.setRoundId(match.getRoundId());
                            rank.setAreaId(match.getAreaId());
                            rank.setSportUserId(oldRecord.getPlayerId());
                            rank.setMatchId(match.getId());
                            rank.setRecordId(record.getId());
                            rank.setDuration(item.getDuration());
                            rank.setCreateTime(item.getFinishTime());
                            seriesRoundRankMapper.insert(rank);
                        } else {
                            rank.setMatchId(record.getMatchId());
                            rank.setRecordId(record.getId());
                            rank.setDuration(item.getDuration());
                            rank.setCreateTime(item.getFinishTime());
                            seriesRoundRankMapper.updateById(rank);
                        }
                    }
                }
                sportUserTrainingLogService.createBicycleTrainingLog(oldRecord.getPlayerId(), GameMode.SERIES_MATCH,
                        match.getId(), sourceName, match.getMap(), match.getPath(), record.getPraiseNum(), item);
            }
        });
    }

}
