package com.antai.sport.http.server.server.api.business.club.service;

import com.antai.sport.http.server.common.exception.BusinessException;
import com.antai.sport.http.server.constants.ClubMemberType;
import com.antai.sport.http.server.constants.DeleteStatus;
import com.antai.sport.http.server.repository.club.entity.Club;
import com.antai.sport.http.server.repository.club.entity.ClubMember;
import com.antai.sport.http.server.repository.club.entity.ClubNotice;
import com.antai.sport.http.server.repository.club.mapper.ClubMapper;
import com.antai.sport.http.server.repository.club.mapper.ClubMemberMapper;
import com.antai.sport.http.server.repository.club.mapper.ClubNoticeMapper;
import com.antai.sport.http.server.server.api.business.club.converter.ClubConverter;
import com.antai.sport.http.server.server.api.business.club.dto.*;
import com.antai.sport.http.server.server.api.business.club.mapper.ClubBusinessMapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;

@Service
public class ClubService {

    @Resource
    private ClubMapper clubMapper;
    @Resource
    private ClubMemberMapper clubMemberMapper;
    @Resource
    private ClubNoticeMapper clubNoticeMapper;

    @Resource
    private ClubBusinessMapper clubBusinessMapper;

    @Resource
    private ClubConverter clubConverter;

    /**
     * 创建俱乐部
     *
     * @param dto
     * @return
     */
    @Transactional
    public Long create(CreateClubReqDTO dto) {
        if (clubBusinessMapper.checkChairman(dto.getCreateUser())) {
            throw new BusinessException("您已经是俱乐部会长，不能创建多个俱乐部");
        }
        checkClubNameRepeat(null, dto.getName());
        Club club = clubConverter.createDTOtoClub(dto, LocalDateTime.now());
        clubMapper.insert(club);
        ClubMember clubMember = new ClubMember();
        clubMember.setClubId(club.getId());
        clubMember.setType(ClubMemberType.CHAIRMAN);
        clubMember.setSportUserId(dto.getCreateUser());
        clubMember.setDeleted(0);
        clubMemberMapper.insert(clubMember);
        return club.getId();
    }

    /**
     * 修改俱乐部信息
     *
     * @param dto
     */
    public void update(UpdateClubReqDTO dto) {
        checkClubNameRepeat(dto.getId(), dto.getName());
        Club club = clubConverter.updateDTOtoClub(dto, LocalDateTime.now());
        clubMapper.updateById(club);
    }

    /**
     * 校验俱乐部名称是否重复
     *
     * @param id
     * @param name
     */
    private void checkClubNameRepeat(Long id, String name) {
        if (clubBusinessMapper.checkClubNameRepeat(id, name)) {
            throw new BusinessException("俱乐部名称重复，请换一个名称试试");
        }
    }

    /**
     * 创建俱乐部通知
     *
     * @param dto
     * @return
     */
    public Long createClubNotice(CreateClubNoticeDTO dto) {
        this.checkClubChairman(dto.getClubId(), dto.getCreateUser());
        ClubNotice notice = clubConverter.createClubNoticeDTOToClubNotice(dto, LocalDateTime.now());
        clubNoticeMapper.insert(notice);
        return notice.getId();
    }

    /**
     * 修改俱乐部通知
     *
     * @param dto
     * @return
     */
    public void updateClubNotice(UpdateClubNoticeDTO dto) {
        this.checkClubChairman(dto.getClubId(), dto.getSportUserId());
        ClubNotice notice = clubConverter.updateClubNoticeDTOToClubNotice(dto, LocalDateTime.now());
        clubNoticeMapper.updateById(notice);
    }

    /**
     * 删除比赛通知
     *
     * @param dto
     */
    public void deleteClubNotice(DeleteClubNoticeDTO dto) {
        this.checkClubChairman(dto.getClubId(), dto.getSportUserId());
        ClubNotice notice = new ClubNotice();
        notice.setId(dto.getId());
        notice.setDeleted(1);
        clubNoticeMapper.updateById(notice);
    }


    /**
     * 校验通知操作用户是否为当前俱乐部会长
     *
     * @param clubId
     * @param sportUserId
     */
    private void checkClubChairman(Long clubId, Long sportUserId) {
        QueryWrapper<ClubMember> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("type", ClubMemberType.CHAIRMAN);
        queryWrapper.eq("deleted", DeleteStatus.UNDELETED);
        queryWrapper.eq("club_id", clubId);
        queryWrapper.eq("sport_user_id", sportUserId);
        ClubMember clubMember = clubMemberMapper.selectOne(queryWrapper);
        if (clubMember == null) {
            throw new BusinessException("您不是当前俱乐部会长，不能进行当前操作");
        }
    }

    /**
     * 加入俱乐部
     *
     * @param dto
     */
    @Transactional
    public void join(JoinOrLeaveClubDTO dto) {
        Club club = clubMapper.selectById(dto.getClubId());
        if (club == null) {
            throw new BusinessException("您要加入的俱乐部不存在");
        }

        this.checkJoin(dto.getSportUserId());

        ClubMember clubMember = new ClubMember();
        clubMember.setClubId(dto.getClubId());
        clubMember.setType(ClubMemberType.CHAIRMAN);
        clubMember.setSportUserId(dto.getSportUserId());
        clubMember.setDeleted(0);
        clubMemberMapper.insert(clubMember);
        this.updateClubMemberNum(dto.getClubId());
    }

    /**
     * 离开俱乐部
     *
     * @param dto
     */
    @Transactional
    public void leave(JoinOrLeaveClubDTO dto) {
        ClubMember clubMember = new ClubMember();
        clubMember.setDeleted(1);
        QueryWrapper<ClubMember> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("deleted", 0);
        queryWrapper.eq("club_id", dto.getClubId());
        queryWrapper.eq("sport_user_id", dto.getSportUserId());
        clubMemberMapper.update(clubMember, queryWrapper);
        this.updateClubMemberNum(dto.getClubId());
    }

    /**
     * 验证用户是否已经加入俱乐部
     */
    private void checkJoin(Long sportUserId) {
        QueryWrapper<ClubMember> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("deleted", 0);
        queryWrapper.eq("sport_user_id", sportUserId);
        ClubMember clubMember = clubMemberMapper.selectOne(queryWrapper);
        if (clubMember != null) {
            throw new BusinessException("您已经加入过俱乐部，不能同时加入多个俱乐部");
        }
    }


    /**
     * 查询俱乐部当前会员总数
     *
     * @param clubId
     * @return
     */
    private void updateClubMemberNum(Long clubId) {
        QueryWrapper<ClubMember> countQuery = new QueryWrapper<>();
        countQuery.eq("club_id", clubId);
        countQuery.eq("deleted", 0);
        Club club = new Club();
        club.setId(clubId);
        club.setMemberNumber(clubMemberMapper.selectCount(countQuery));
        clubMapper.updateById(club);
    }

    /**
     * 获取俱乐部列表
     *
     * @param dto
     * @return
     */
    public List<ClubRespDTO> getList(ClubListReqDTO dto) {
        Page<ClubRespDTO> pageParam = new Page<>(dto.getPageNo(), dto.getPageSize());
        return clubBusinessMapper.getClubList(pageParam, dto).getRecords();
    }

    /**
     * 获取玩家俱乐部信息
     *
     * @param sportUserId
     * @return
     */
    public PlayerClubHomeDTO getPlayerClubHome(Long sportUserId) {
        QueryWrapper<ClubMember> memberQuery = new QueryWrapper<>();
        memberQuery.eq("deleted", 0);
        memberQuery.eq("sport_user_id", sportUserId);
        ClubMember clubMember = clubMemberMapper.selectOne(memberQuery);
        if (clubMember == null) {
            return null;
        }
        PlayerClubHomeDTO result = new PlayerClubHomeDTO();
        result.setClub(clubBusinessMapper.getClubInfo(clubMember.getClubId(), null));
        result.setMemberList(clubBusinessMapper.getClubMemberList(clubMember.getClubId()));
        result.setNoticeList(clubBusinessMapper.getClubNotice(clubMember.getClubId()));
        return result;
    }

    /**
     * 加载俱乐部主页
     * @param clubId
     * @return
     */
    public ClubHomeDTO getClubHome(Long clubId) {
        ClubHomeDTO result = new ClubHomeDTO();
        result.setClub(clubBusinessMapper.getClubInfo(clubId, null));
        result.setMemberList(clubBusinessMapper.getClubMemberList(clubId));
        return result;
    }

}
