package com.antai.sport.http.server.management.api.business.series.service;

import com.antai.sport.http.server.common.exception.BusinessException;
import com.antai.sport.http.server.management.api.business.series.converter.SeriesConverter;
import com.antai.sport.http.server.management.api.business.series.dto.*;
import com.antai.sport.http.server.management.api.business.series.mapper.SeriesBusinessMapper;
import com.antai.sport.http.server.management.api.business.series.vo.*;
import com.antai.sport.http.server.management.api.business.simplematch.dto.vo.MapPathVO;
import com.antai.sport.http.server.repository.map.entity.Map;
import com.antai.sport.http.server.repository.map.entity.MapPath;
import com.antai.sport.http.server.repository.map.mapper.MapMapper;
import com.antai.sport.http.server.repository.map.mapper.MapPathMapper;
import com.antai.sport.http.server.repository.series.entity.*;
import com.antai.sport.http.server.repository.series.mapper.*;
import com.antai.sport.http.server.repository.sys.entity.SysUser;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.metadata.OrderItem;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.google.common.collect.Iterables;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

@Service
public class SeriesService {
    @Resource
    private SeriesBusinessMapper seriesBusinessMapper;
    @Resource
    private SeriesMapper seriesMapper;
    @Resource
    private SeriesAreaMapper seriesAreaMapper;
    @Resource
    private SeriesRoundMapper seriesRoundMapper;
    @Resource
    private MapMapper mapMapper;
    @Resource
    private MapPathMapper mapPathMapper;
    @Resource
    private SeriesMatchMapper seriesMatchMapper;
    @Resource
    private SeriesMatchPlayerRecordMapper seriesMatchPlayerRecordMapper;
    @Resource
    private SeriesMatchFinalistsMapper seriesMatchFinalistsMapper;

    @Resource
    private SeriesConverter seriesConverter;

    public SeriesListPageVO getSeriesList(SeriesListQueryDTO dto) {
        SeriesListPageVO result = new SeriesListPageVO();
        Page<SeriesListVO> pageParam = new Page<>(dto.getPageNo(), dto.getPageSize());
        pageParam.addOrder(new OrderItem("id", false));
        IPage<SeriesListVO> seriesListPage = seriesBusinessMapper.getSeriesList(pageParam, dto);
        result.setPageNo(dto.getPageNo());
        result.setTotalCount(seriesListPage.getTotal());
        result.setData(seriesListPage.getRecords());
        return result;
    }

    public Series getSeries(Long id) {
        Series series;
        if (id == null || id == 0) {
            series = new Series();
            series.setStatus(10);
            series.setApplyNum(0);

        } else {
            series = seriesMapper.selectById(id);
            if (series == null) {
                throw new BusinessException("您查询的系列赛不存在");
            }
        }
        return series;
    }

    public void saveSeries(SeriesSaveDTO dto, SysUser loginUser) {
        Series series = seriesConverter.toSeries(dto);
        if (series.getId() != null && series.getId() > 0) {
            series.setUpdateBy(loginUser.getId());
            series.setUpdateTime(LocalDateTime.now());
            seriesMapper.updateById(series);
        } else {
            series.setStatus(10);
            series.setApplyNum(0);
            series.setFinished(false);
            series.setCreateBy(loginUser.getId());
            series.setCreateTime(LocalDateTime.now());
            seriesMapper.insert(series);
        }
    }

    public void publishSeries(Long id) {
        Series series = seriesMapper.selectById(id);
        if (series.getStatus().equals(10)) {
            series.setStatus(20);
        } else {
            series.setStatus(10);
        }
        seriesMapper.updateById(series);
    }

    public void finishedSeries(Long id) {
        Series series = seriesMapper.selectById(id);
        if (series.getFinished()) {
            series.setFinished(false);
        } else {
            series.setFinished(true);
        }
        seriesMapper.updateById(series);
    }

    public void deletedSeries(Long id) {
        Series series = seriesMapper.selectById(id);
        series.setDeleted(1);
        seriesMapper.updateById(series);
    }

    public List<SeriesAreaDTO> getSeriesAreaList(Long seriesId) {
        QueryWrapper<SeriesArea> areaQuery = new QueryWrapper<>();
        areaQuery.lambda().eq(SeriesArea::getSeriesId, seriesId).eq(SeriesArea::getDeleted, 0);
        List<SeriesArea> areaList = seriesAreaMapper.selectList(areaQuery);
        return seriesConverter.toSeriesAreaDTO(areaList);
    }

    @Transactional
    public void saveSeriesArea(SeriesAreaSaveDTO dto) {
        List<SeriesArea> areaList = seriesConverter.toSeriesArea(dto.getAreaList());
        LocalDateTime now = LocalDateTime.now();
        for (SeriesArea item : areaList) {
            if (item.getId() > 0) {
                item.setUpdateTime(now);
                seriesAreaMapper.updateById(item);
            } else {
                item.setId(null);
                item.setSeriesId(dto.getSeriesId());
                item.setCreateTime(now);
                seriesAreaMapper.insert(item);
            }
        }

        for (Long id : dto.getDeleteList()) {
            SeriesArea area = seriesAreaMapper.selectById(id);
            if (area != null) {
                area.setDeleted(1);
                seriesAreaMapper.updateById(area);
            }
        }
    }

    public SeriesRoundDetailVO getSeriesRoundDetail(Long roundId) {
        SeriesRoundDetailVO result = new SeriesRoundDetailVO();
        SeriesRound round;
        if (roundId == null || roundId == 0) {
            round = new SeriesRound();
            round.setStatus(10);
            round.setFinished(false);
            round.setStarted(false);
            round.setResultConfirm(false);
        } else {
            round = seriesRoundMapper.selectById(roundId);
            if (round == null) {
                throw new BusinessException("您查询的系列赛不存在");
            }
        }
        result.setSeriesRound(round);

        List<Map> mapList = mapMapper.selectList(null);
        List<MapPath> mapPathList = mapPathMapper.selectList(null);
        HashMap<String, MapPathVO> mapPathVOHashMap = new HashMap<>();
        List<MapPathVO> mapPathVOList = new ArrayList<>(mapList.size());
        mapList.forEach(item -> {
            MapPathVO mapVO = new MapPathVO();
            mapVO.setId(item.getId());
            mapVO.setKey(item.getMapKey());
            mapVO.setName(item.getMapName());
            mapVO.setPathList(new ArrayList<>());
            mapPathVOList.add(mapVO);
            mapPathVOHashMap.put(item.getMapKey(), mapVO);
        });
        mapPathList.forEach(item -> {
            MapPathVO mapPathVO = new MapPathVO();
            mapPathVO.setId(item.getId());
            mapPathVO.setKey(item.getPathKey());
            mapPathVO.setName(item.getPathName());
            if (mapPathVOHashMap.containsKey(item.getMapKey())) {
                mapPathVOHashMap.get(item.getMapKey()).getPathList().add(mapPathVO);
            }
        });
        result.setMapList(mapPathVOList);

        return result;
    }

    @Transactional
    public void saveSeriesRound(SeriesRoundSaveDTO dto) {
        SeriesRound seriesRound = seriesConverter.toSeriesRound(dto);
        List<SeriesRound> roundList = seriesBusinessMapper.getRoundBySeriesId(dto.getSeriesId());
        if (seriesRound.getId() == null || seriesRound.getId() == 0) {
            seriesRound.setFinalRound(true);
            seriesRound.setShowOrder(1);
            seriesRound.setStatus(10);
            if (!roundList.isEmpty()) {
                if (seriesRound.getType().equals(10)) {
                    throw new BusinessException("只有第一轮可以配置为预选赛");
                }
                SeriesRound lastRound = Iterables.getLast(roundList);
                lastRound.setFinalRound(false);
                seriesRoundMapper.updateById(lastRound);
                seriesRound.setPreRoundId(lastRound.getId());
                seriesRound.setShowOrder(lastRound.getShowOrder() + 1);
            }
            seriesRoundMapper.insert(seriesRound);
        } else {
            if (seriesRound.getType().equals(10) && !roundList.get(0).getId().equals(seriesRound.getId())) {
                throw new BusinessException("只有第一轮可以配置为预选赛");
            }
            seriesRoundMapper.updateById(seriesRound);
            QueryWrapper<SeriesMatch> matchQuery = new QueryWrapper<>();
            matchQuery.lambda().eq(SeriesMatch::getRoundId, seriesRound.getId()).eq(SeriesMatch::getDeleted, 0);
            List<SeriesMatch> matchList = seriesMatchMapper.selectList(matchQuery);
            for (SeriesMatch item : matchList) {
                item.setMap(seriesRound.getMap());
                item.setPath(seriesRound.getPath());
                item.setDistance(seriesRound.getDistance());
                item.setClimbDistance(seriesRound.getClimbDistance());
                item.setType(seriesRound.getType());
                seriesMatchMapper.updateById(item);
            }
        }
    }

    public List<SeriesRoundListVO> getRoundListBySeriesId(Long seriesId) {
        return seriesBusinessMapper.getRoundListBySeriesId(seriesId);
    }

    @Transactional
    public void deleteRound(Long roundId) {
        SeriesRound seriesRound = seriesRoundMapper.selectById(roundId);
        if (!seriesRound.getFinalRound()) {
            throw new BusinessException("当前轮次不是最后一轮，请从最后一轮开始删除");
        }
        seriesRound.setDeleted(1);
        seriesRoundMapper.updateById(seriesRound);
        if (seriesRound.getPreRoundId() != null) {
            SeriesRound preRound = seriesRoundMapper.selectById(seriesRound.getPreRoundId());
            preRound.setFinalRound(true);
            seriesRoundMapper.updateById(preRound);
        }
    }

    @Transactional
    public void changeRoundStatus(Long roundId) {
        SeriesRound round = seriesRoundMapper.selectById(roundId);
        if (round.getStatus().equals(10)) {
            round.setStatus(20);
        } else {
            round.setStatus(10);
        }
        seriesRoundMapper.updateById(round);
    }

    public List<SeriesMatchListVO> getMatchByRoundId(Long roundId) {
        return seriesBusinessMapper.getMatchByRoundId(roundId);
    }

    public void changeMatchStatus(Long matchId) {
        SeriesMatch match = seriesMatchMapper.selectById(matchId);
        if (match.getStatus().equals(10)) {
            match.setStatus(20);
        } else {
            match.setStatus(10);
        }
        seriesMatchMapper.updateById(match);
    }

    public void deleteMatch(Long matchId) {
        SeriesMatch match = seriesMatchMapper.selectById(matchId);
        match.setDeleted(1);
        seriesMatchMapper.updateById(match);
    }

    public SeriesMatchDetailVO getSeriesMatchDetail(Long roundId, Long matchId) {
        SeriesMatchDetailVO result = new SeriesMatchDetailVO();
        SeriesRound seriesRound = seriesRoundMapper.selectById(roundId);
        QueryWrapper<SeriesArea> areaQuery = new QueryWrapper<>();
        areaQuery.lambda().eq(SeriesArea::getSeriesId, seriesRound.getSeriesId()).eq(SeriesArea::getDeleted, 0);
        List<SeriesArea> areaList = seriesAreaMapper.selectList(areaQuery);
        result.setAreaList(areaList);
        SeriesMatch seriesMatch;
        if (matchId == 0) {
            seriesMatch = new SeriesMatch();
            seriesMatch.setStatus(10);
        } else {
            seriesMatch = seriesMatchMapper.selectById(matchId);
        }
        result.setMatch(seriesMatch);
        return result;
    }

    public void saveMatch(SeriesMatchSaveDTO dto) {
        SeriesMatch match = seriesConverter.toSeriesMatch(dto);
        if (match.getId() == 0) {
            SeriesRound round = seriesRoundMapper.selectById(dto.getRoundId());
            match.setSeriesId(round.getSeriesId());
            match.setType(round.getType());
            match.setMap(round.getMap());
            match.setPath(round.getPath());
            match.setDistance(round.getDistance());
            match.setClimbDistance(round.getClimbDistance());
            match.setStatus(10);
            seriesMatchMapper.insert(match);
        } else {
            seriesMatchMapper.updateById(match);
        }
    }

    public SeriesQualifyingMatchRankPageVO getSeriesQualifyingMatchRank(SeriesQualifyingMatchRankDTO dto) {
        SeriesQualifyingMatchRankPageVO result = new SeriesQualifyingMatchRankPageVO();
        result.setPageNo(dto.getPageNo());
        Page<SeriesQualifyingMatchRankVO> pageParam = new Page<>(dto.getPageNo(), dto.getPageSize());
        pageParam.addOrder(new OrderItem("t1.duration", true));
        pageParam.addOrder(new OrderItem("t1.create_time", true));
        pageParam.addOrder(new OrderItem("t1.id", true));
        IPage<SeriesQualifyingMatchRankVO> rankPage = seriesBusinessMapper
                .getSeriesQualifyingMatchRank(pageParam, dto.getRoundId(), dto.getAreaId());
        result.setData(rankPage.getRecords());
        result.setTotalCount(rankPage.getTotal());
        return result;
    }

    public void changeMatchRecordStatus(Long recordId, Integer status) {
        SeriesMatchPlayerRecord record = seriesMatchPlayerRecordMapper.selectById(recordId);
        record.setResultConfirmStatus(status);
        seriesMatchPlayerRecordMapper.updateById(record);
    }

    public SeriesQualifyingMatchRankPageVO getKnockoutMatchRecord(SeriesKnockoutMatchRecordDTO dto) {
        SeriesQualifyingMatchRankPageVO result = new SeriesQualifyingMatchRankPageVO();
        result.setPageNo(dto.getPageNo());
        Page<SeriesQualifyingMatchRankVO> pageParam = new Page<>(dto.getPageNo(), dto.getPageSize());
        pageParam.addOrder(new OrderItem("t2.status", false));
        pageParam.addOrder(new OrderItem("t2.match_rank", true));
        pageParam.addOrder(new OrderItem("t2.distance", false));
        pageParam.addOrder(new OrderItem("t2.duration", true));
        pageParam.addOrder(new OrderItem("t2.id", true));
        IPage<SeriesQualifyingMatchRankVO> rankPage = seriesBusinessMapper
                .getKnockoutMatchRecord(pageParam, dto.getMatchId());
        result.setData(rankPage.getRecords());
        result.setTotalCount(rankPage.getTotal());
        return result;
    }

    public SeriesMatchFinalistsPageVO getSeriesMatchFinalists(SeriesMatchFinalistsPageDTO dto) {
        SeriesMatchFinalistsPageVO result = new SeriesMatchFinalistsPageVO();
        result.setPageNo(dto.getPageNo());
        SeriesMatch match = seriesMatchMapper.selectById(dto.getMatchId());
        SeriesRound round = seriesRoundMapper.selectById(match.getRoundId());
        Page<SeriesMatchFinalistsVO> pageParam = new Page<>(dto.getPageNo(), dto.getPageSize());
        pageParam.addOrder(new OrderItem("t1.id", false));
        IPage<SeriesMatchFinalistsVO> finalistsPage = seriesBusinessMapper
                .getSeriesMatchFinalists(pageParam, match.getSeriesId(), dto.getMatchId(), round.getPreRoundId());
        result.setData(finalistsPage.getRecords());
        result.setTotalCount(finalistsPage.getTotal());
        return result;
    }

    public SeriesMatchFinalistsPageVO getKnockoutMatchCandidate(SeriesMatchFinalistsPageDTO dto) {
        SeriesMatchFinalistsPageVO result = new SeriesMatchFinalistsPageVO();
        result.setPageNo(dto.getPageNo());
        SeriesMatch match = seriesMatchMapper.selectById(dto.getMatchId());
        SeriesRound round = seriesRoundMapper.selectById(match.getRoundId());
        Page<SeriesMatchFinalistsVO> pageParam = new Page<>(dto.getPageNo(), dto.getPageSize());

        IPage<SeriesMatchFinalistsVO> candidatePage;
        if (round.getPreRoundId() == null) {
            pageParam.addOrder(new OrderItem("t1.id", true));
            candidatePage = seriesBusinessMapper
                    .getCandidateWithoutQualifying(pageParam, round.getSeriesId(), round.getId());
        } else {
            pageParam.addOrder(new OrderItem("t1.duration", true));
            pageParam.addOrder(new OrderItem("t1.id", true));
            candidatePage = seriesBusinessMapper
                    .getCandidate(pageParam, match.getSeriesId(), round.getPreRoundId(), round.getId());
        }
        result.setData(candidatePage.getRecords());
        result.setTotalCount(candidatePage.getTotal());
        return result;
    }

    public void addToFinalists(Long matchId, Long sportUserId) {
        SeriesMatch match = seriesMatchMapper.selectById(matchId);
        QueryWrapper<SeriesMatchFinalists> finalistsQuery = new QueryWrapper<>();
        finalistsQuery.lambda().eq(SeriesMatchFinalists::getRoundId, match.getRoundId())
                .eq(SeriesMatchFinalists::getSportUserId, sportUserId);
        Integer count = seriesMatchFinalistsMapper.selectCount(finalistsQuery);
        if (count > 0) {
            throw new BusinessException("当前用户已经加入到当前轮次当中，不能重复添加");
        }
        SeriesMatchFinalists finalists = new SeriesMatchFinalists();
        finalists.setMatchId(matchId);
        finalists.setRoundId(match.getRoundId());
        finalists.setSportUserId(sportUserId);
        seriesMatchFinalistsMapper.insert(finalists);
    }

    public void deleteFinalists(Long id) {
        seriesMatchFinalistsMapper.deleteById(id);
    }

}
