package com.antai.sport.http.server.common.config;

import io.swagger.models.auth.In;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.service.*;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spi.service.contexts.SecurityContext;
import springfox.documentation.spring.web.plugins.ApiSelectorBuilder;
import springfox.documentation.spring.web.plugins.Docket;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

@Configuration
@ConfigurationProperties(prefix = "swagger")
public class SwaggerConfig {
    private static final Logger logger = LoggerFactory.getLogger(SwaggerConfig.class);

    private boolean enable;
    private String projectName;
    private String scanPackage;
    private String headerTokenKey;
    private String apiVersion;
    private String author;
    private String email;
    private String description;
    private String serviceWebsite;
    private List<String> ignoredParameterList;

    @Bean
    public Docket createRestApi() {

        Docket docket = new Docket(DocumentationType.OAS_30)
                .pathMapping("/")
                .enable(enable)
                .apiInfo(apiInfo())
                .groupName(projectName)
                .useDefaultResponseMessages(false)
                .securityContexts(initSecurityContext())
                .securitySchemes(initSecuritySchemes());

        if (ignoredParameterList != null) {
            List<Class<?>> ignoreClassList = new ArrayList<>(ignoredParameterList.size());
            for (String classPath : ignoredParameterList) {
                try {
                    ignoreClassList.add(Class.forName(classPath));
                } catch (Exception ex) {
                    logger.error(classPath + "类解析异常");
                }
            }
            if (!ignoreClassList.isEmpty()) {
                Class<?>[] ignoreClassArray = new Class[ignoreClassList.size()];
                ignoreClassList.toArray(ignoreClassArray);
                docket.ignoredParameterTypes(ignoreClassArray);
            }
        }

        ApiSelectorBuilder builder = docket.select()
                .apis(RequestHandlerSelectors.basePackage(scanPackage))
                .paths(PathSelectors.any());

        return builder.build();
    }

    private ApiInfo apiInfo() {
        return new ApiInfoBuilder()
                .title(projectName)
                .description(description)
                .termsOfServiceUrl(serviceWebsite)
                .contact(new Contact(author, "", email))
                .version(apiVersion)
                .build();
    }

    private List<SecurityScheme> initSecuritySchemes() {
        ApiKey apiKey = new ApiKey(headerTokenKey, headerTokenKey, In.HEADER.toValue());
        return Collections.singletonList(apiKey);
    }

    private List<SecurityContext> initSecurityContext() {
        return Collections.singletonList(
                SecurityContext.builder()
                        .securityReferences(
                                Collections.singletonList(
                                        SecurityReference.builder()
                                                .reference(headerTokenKey)
                                                .scopes(new AuthorizationScope[]{new AuthorizationScope("global", "accessEverything")})
                                                .build()
                                )
                        )
                        .build()
        );
    }

    public void setEnable(boolean enable) {
        this.enable = enable;
    }

    public void setProjectName(String projectName) {
        this.projectName = projectName;
    }

    public void setScanPackage(String scanPackage) {
        this.scanPackage = scanPackage;
    }

    public void setHeaderTokenKey(String headerTokenKey) {
        this.headerTokenKey = headerTokenKey;
    }

    public void setApiVersion(String apiVersion) {
        this.apiVersion = apiVersion;
    }

    public void setAuthor(String author) {
        this.author = author;
    }

    public void setEmail(String email) {
        this.email = email;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public void setServiceWebsite(String serviceWebsite) {
        this.serviceWebsite = serviceWebsite;
    }

    public void setIgnoredParameterList(List<String> ignoredParameterList) {
        this.ignoredParameterList = ignoredParameterList;
    }
}
