import type { BaseModel, PageQuery } from '#/api/baseModel';

import { commonExport } from '#/api/helper';
import { requestClient } from '#/api/request';

export namespace EmployeeFlowApi {
  export interface EmployeeFlow extends BaseModel {
    /**
     * 申请编号
     */
    applyCode?: string;
    /**
     * 审批类型
     */
    flowType?: string;
    /**
     * 员工信息ID
     */
    employeeId?: number;

    /**
     * 开始时间
     */
    startDate?: string;
    /**
     * 结束时间
     */
    endDate?: string;
    /**
     * 状态
     */
    status?: string;
    /**
     * 备注
     */
    remark?: string;
  }

  export interface EmployeeFlowVo {
    /**
     * 序号
     */
    id?: number;
    /**
     * 申请编号
     */
    applyCode?: string;
    /**
     * 审批类型
     */
    flowType?: string;
    /**
     * 员工信息ID
     */
    employeeId?: number;

    /**
     * 审计内容
     */
    auditLogList?: Array<AuditLogVo>;
    /**
     * 状态
     */
    status?: string;
    /**
     * 备注
     */
    remark?: string;
  }

  export interface AuditLogVo {
    /**
     * 审计名称（入职、调职、离职）
     */
    auditName?: string;
    /**
     * 字段名称
     */
    auditField?: string;
    /**
     * 字段注释名称
     */
    auditFieldName?: string;
    /**
     * 变更前值
     */
    beforeVal?: string;
    /**
     * 变更后值
     */
    afterVal?: string;
  }
}

/**
 * 查询人事审批对象列表
 * @param params
 * @returns {*} page
 */
export function apiPage(params: PageQuery) {
  return requestClient.get('/employee/flow/page', { params });
}
/**
 * 查询人事审批对象详细
 * @param id
 */
export function apiDetail(id: number) {
  return requestClient.get(`/employee/flow/${id}`);
}
/**
 * 新增人事审批对象
 * @param data
 */
export function apiAdd(data: EmployeeFlowApi.EmployeeFlow) {
  return requestClient.post('/employee/flow', data);
}
/**
 * 修改人事审批对象
 * @param data
 */
export function apiUpdate(data: EmployeeFlowApi.EmployeeFlow) {
  return requestClient.put('/employee/flow', data);
}
/**
 * 删除人事审批对象
 * @param id
 */
export function apiDelete(id: Array<number> | number) {
  return requestClient.delete(`/employee/flow/${id}`);
}
/**
 * 导出人事审批对象
 * @param params
 */
export function apiExport(params: PageQuery) {
  return commonExport('/employee/flow/export', params);
}

/**
 * 提交 & 发起流程(后端发起)
 * @param data data
 * @returns void
 */
export function submitAndStartWorkflow(data: EmployeeFlowApi.EmployeeFlow) {
  return requestClient.post('/employee/flow/submitAndFlowStart', data, {
    successMessageMode: 'message',
  });
}
