import type { AxiosRequestConfig } from '@vben/request';

import type { BaseModel, ID, IDS, PageQuery } from '#/api/baseModel';

import { ContentTypeEnum } from '#/api/helper';
import { requestClient } from '#/api/request';

export interface OssFile extends BaseModel {
  /**
   * 文件名
   */
  fileName?: string;
  /**
   * 原名
   */
  originalName?: string;
  /**
   * 文件后缀名
   */
  fileSuffix?: string;
  /**
   * URL地址
   */
  url?: string;
  /**
   * 扩展字段
   */
  ext1?: string;
  /**
   * 服务商
   */
  service?: string;
}

/**
 * 文件list
 * @param params 参数
 * @returns 分页
 */
export function ossList(params?: PageQuery) {
  return requestClient.get('/system/oss/page', { params });
}

/**
 * 查询文件信息 返回为数组
 * @param ossIds id数组`
 * @returns 信息数组
 */
export function ossInfo(ossIds: ID | IDS) {
  return requestClient.get(`/system/oss/listByIds/${ossIds}`);
}

/**
 * @deprecated 使用apps/web-antd/src/api/core/upload.ts uploadApi方法
 * @param file 文件
 * @returns void
 */
export function ossUpload(file: Blob | File) {
  const formData = new FormData();
  formData.append('file', file);
  return requestClient.post('/system/oss/upload', formData, {
    headers: { 'Content-Type': ContentTypeEnum.FORM_DATA },
    timeout: 30 * 1000,
    successMessageMode: 'message',
  });
}

/**
 * 下载文件  返回为二进制
 * @param ossId ossId
 * @param onDownloadProgress 下载进度(可选)
 * @returns blob
 */
export function ossDownload(
  ossId: ID,
  onDownloadProgress?: AxiosRequestConfig['onDownloadProgress'],
) {
  return requestClient.get<Blob>(`/system/oss/download/${ossId}`, {
    responseType: 'blob',
    timeout: 30 * 1000,
    isTransformResponse: false,
    onDownloadProgress,
  });
}

/**
 * 在使用浏览器原生下载前检测是否登录
 * 这里的方案为请求一次接口 如果登录超时会走到response的401逻辑
 * 如果没有listByIds的权限 也不会弹出无权限提示
 * 仅仅是为了检测token是否有效使用
 *
 * @returns void
 */
export function checkLoginBeforeDownload() {
  return requestClient.get<OssFile[]>(`/system/oss/1`, {
    errorMessageMode: 'none',
  });
}

/**
 * 删除文件
 * @param ossIds id数组
 * @returns void
 */
export function ossRemove(ossIds: ID | IDS) {
  return requestClient.delete(`/system/oss/${ossIds}`, {
    successMessageMode: 'message',
  });
}
