import type { BaseModel, PageQuery } from '#/api/baseModel';

import { commonExport } from '#/api/helper';
import { requestClient } from '#/api/request';

export namespace InterviewProfileApi {
  export interface InterviewProfile extends BaseModel {
    /**
     * 员工信息ID
     */
    employeeId?: number;

    /**
     * 招揽岗位
     */
    jobPosition?: string;
    /**
     * 姓名
     */
    name?: string;
    /**
     * 性别
     */
    gender?: string;
    /**
     * 身份证号
     */
    idCardNumber?: number;

    /**
     * 学历
     */
    education?: string;
    /**
     * 院校
     */
    school?: string;
    /**
     * 专业
     */
    major?: string;
    /**
     * 联系方式
     */
    phoneNumber?: string;
    /**
     * 标杆企业
     */
    benchmarkEnterprise?: string;
    /**
     * 面试时间
     */
    interviewDate?: string;
    /**
     * 面试情况
     */
    interviewSituation?: string;
    /**
     * 面试结果
     */
    interviewResult?: string;
    /**
     * 备注
     */
    remark?: string;
  }
}

/**
 * 查询面试档案列表
 * @param params
 * @returns {*} page
 */
export function apiPage(params: PageQuery) {
  return requestClient.get('/employee/interviewProfile/page', { params });
}
/**
 * 查询面试档案详细
 * @param id
 */
export function apiDetail(id: number) {
  return requestClient.get(`/employee/interviewProfile/${id}`);
}
/**
 * 新增面试档案
 * @param data
 */
export function apiAdd(data: InterviewProfileApi.InterviewProfile) {
  return requestClient.post('/employee/interviewProfile', data);
}
/**
 * 修改面试档案
 * @param data
 */
export function apiUpdate(data: InterviewProfileApi.InterviewProfile) {
  return requestClient.put('/employee/interviewProfile', data);
}
/**
 * 删除面试档案
 * @param id
 */
export function apiDelete(id: Array<number> | number) {
  return requestClient.delete(`/employee/interviewProfile/${id}`);
}
/**
 * 导出面试档案
 * @param params
 */
export function apiExport(params: PageQuery) {
  return commonExport('/employee/interviewProfile/export', params);
}

/**
 * 下载用户导入模板
 * @returns blob
 */
export function downloadImportTemplate() {
  return commonExport('/employee/interviewProfile/importTemplate', {});
}

/**
 * 从excel导入面试信息
 * @param file
 * @returns void
 */
export function interviewProfileImportData(file: Blob) {
  return requestClient.post<{ errMessage: string; success: boolean }>(
    '/employee/interviewProfile/import',
    { file },
    {
      headers: {
        'Content-Type': 'multipart/form-data;charset=UTF-8',
      },
      isTransformResponse: false,
      responseReturn: 'body',
      timeout: 300_000,
    },
  );
}
