import { z } from '#/adapter/form';

// 校验码计算和验证函数
function verifyIDCardChecksum(idNumber: string): boolean {
  if (idNumber.length !== 18) return false;

  // 1. 权重因子 [citation:5][citation:6][citation:8]
  const weights = [7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2];
  // 2. 校验码对应表 [citation:5][citation:6][citation:8]
  const checkCodes = ['1', '0', 'X', '9', '8', '7', '6', '5', '4', '3', '2'];

  let sum = 0;
  for (let i = 0; i < 17; i++) {
    sum += parseInt(idNumber[i], 10) * weights[i];
  }

  const mod = sum % 11;
  const expectedCheckCode = checkCodes[mod];

  // 3. 检查最后一位校验码是否匹配（包括大小写X的处理）
  return idNumber[17].toUpperCase() === expectedCheckCode;
}

// 定义并导出Zod模式
export const idCardSchema = z.string()
  .min(1, { message: "身份证号不能为空" })
  // 基础正则校验：基本格式和日期粗略校验
  .regex(/^[1-9]\d{5}(18|19|20)\d{2}(0[1-9]|1[0-2])(0[1-9]|[12]\d|3[01])\d{3}[\dXx]$/, {
    message: "身份证号码格式不正确"
  })
  // 自定义 refine 校验：校验码精确校验
  .refine((id) => verifyIDCardChecksum(id), {
    message: "身份证号码校验码错误"
  });
