import type { BaseModel, PageQuery } from '#/api/baseModel';

import { commonExport } from '#/api/helper';
import { requestClient } from '#/api/request';

export namespace PostApi {
  export interface Post extends BaseModel {
    /**
     * 部门ID
     */
    deptId?: number;

    /**
     * 岗位编码
     */
    postCode?: string;
    /**
     * 岗位类别编码
     */
    postCategory?: string;
    /**
     * 岗位名称
     */
    postName?: string;
    /**
     * 显示顺序
     */
    postSort?: number;

    /**
     * 状态（1：正常，0：停用）
     */
    status?: number;

    /**
     * 创建部门
     */
    createDept?: number;

    /**
     * 备注
     */
    remark?: string;
  }
}

/**
 * 查询岗位信息列表
 * @param params
 * @returns {*} page
 */
export function apiPage(params: PageQuery) {
  return requestClient.get('/system/post/page', { params });
}
/**
 * 查询岗位信息详细
 * @param id
 */
export function apiDetail(id: number) {
  return requestClient.get(`/system/post/${id}`);
}
/**
 * 新增岗位信息
 * @param data
 */
export function apiAdd(data: PostApi.Post) {
  return requestClient.post('/system/post', data);
}
/**
 * 修改岗位信息
 * @param data
 */
export function apiUpdate(data: PostApi.Post) {
  return requestClient.put('/system/post', data);
}
/**
 * 删除岗位信息
 * @param id
 */
export function apiDelete(id: Array<number> | number) {
  return requestClient.delete(`/system/post/${id}`);
}
/**
 * 导出岗位信息
 * @param params
 */
export function apiExport(params: PageQuery) {
  return commonExport('/system/post/export', params);
}
/**
 * 根据部门id获取岗位下拉列表
 * @param deptId 部门id
 * @returns 岗位
 */
export function postOptionSelect(deptId: number) {
  return requestClient.get<PostApi.Post[]>('/system/post/optionSelect', {
    params: { deptId },
  });
}

/**
 * 岗位专用 - 获取部门树
 * @returns 部门树
 */
export function postDeptTreeSelect() {
  return requestClient.get('/system/post/deptTree');
}
