/**
 * 该文件可自行根据业务逻辑进行调整
 */
import type { RequestClientOptions } from '@vben/request';

import { useAppConfig } from '@vben/hooks';
import { preferences } from '@vben/preferences';
import {
  authenticateResponseInterceptor,
  defaultResponseInterceptor,
  errorMessageResponseInterceptor,
  RequestClient,
} from '@vben/request';
import { useAccessStore } from '@vben/stores';

import { message, Modal } from 'ant-design-vue';

import { useAuthStore } from '#/store';

import { refreshTokenApi } from './core';

const { apiURL, clientId } = useAppConfig(
  import.meta.env,
  import.meta.env.PROD,
);

/**
 * 是否已经处在登出过程中了 一个标志位
 * 主要是防止一个页面会请求多个api 都401 会导致登出执行多次
 */
const isLogoutProcessing = false;

function createRequestClient(baseURL: string, options?: RequestClientOptions) {
  const client = new RequestClient({
    ...options,
    baseURL,
  });

  /**
   * 重新认证逻辑
   */
  async function doReAuthenticate() {
    console.warn('Access token or refresh token is invalid or expired. ');
    const accessStore = useAccessStore();
    const authStore = useAuthStore();
    accessStore.setAccessToken(null);
    if (
      preferences.app.loginExpiredMode === 'modal' &&
      accessStore.isAccessChecked
    ) {
      accessStore.setLoginExpired(true);
    } else {
      await authStore.logout();
    }
  }

  /**
   * 刷新token逻辑
   */
  async function doRefreshToken() {
    const accessStore = useAccessStore();
    const resp = await refreshTokenApi();
    const newToken = resp.data;
    accessStore.setAccessToken(newToken);
    return newToken;
  }

  function formatToken(token: null | string) {
    return token ? `Bearer ${token}` : null;
  }

  // 请求头处理
  client.addRequestInterceptor({
    fulfilled: async (config) => {
      const accessStore = useAccessStore();

      config.headers.Authorization = formatToken(accessStore.accessToken);
      config.headers['Accept-Language'] = preferences.app.locale;
      config.headers.ClientID = clientId;
      return config;
    },
  });

  // 处理返回的响应数据格式
  client.addResponseInterceptor(
    defaultResponseInterceptor({
      codeField: 'success',
      dataField: (responseData, config) => {
        const successMsg = '操作成功';
        if (config.successMessageMode === 'modal') {
          Modal.success({
            content: successMsg,
            title: '成功提示',
          });
        } else if (config.successMessageMode === 'message') {
          message.success(successMsg);
        }
        return responseData.totalCount ? responseData : responseData.data;
      },
      successCode: (success) => {
        return success;
      },
    }),
  );

  // token过期的处理
  client.addResponseInterceptor(
    authenticateResponseInterceptor({
      client,
      doReAuthenticate,
      doRefreshToken,
      enableRefreshToken: preferences.app.enableRefreshToken,
      formatToken,
    }),
  );

  // 通用的错误处理,如果没有进入上面的错误处理逻辑，就会进入这里
  client.addResponseInterceptor(
    errorMessageResponseInterceptor((msg: string, error) => {
      // 这里可以根据业务进行定制,你可以拿到 error 内的信息进行定制化处理，根据不同的 code 做不同的提示，而不是直接使用 message.error 提示 msg
      // 当前mock接口返回的错误字段是 error 或者 message
      const responseData = error?.response?.data ?? {};
      const errorMessage = responseData?.errMessage ?? '';
      console.error('[666]', msg);
      if (!responseData.errCode || responseData.errCode !== '401') {
        // 如果没有错误信息，则会根据状态码进行提示
        message.error(errorMessage || msg);
      }
    }),
  );

  // client.addResponseInterceptor({
  //   fulfilled: async (response) => {
  //     console.log('[customResponseInterceptor]', response);
  //     const axiosResponseData = response.data;
  //     if (
  //       !response.config.responseType ||
  //       response.config.responseType !== 'blob'
  //     ) {
  //       return axiosResponseData;
  //     }
  //     const { errCode, data, errMessage, success } = axiosResponseData;
  //     console.log('[axiosResponseData]', errCode);
  //     switch (errCode) {
  //       case '401': {
  //         // 已经在登出过程中 不再执行
  //         if (isLogoutProcessing) {
  //           throw new Error('登陆过期，请重新登录');
  //         }
  //         isLogoutProcessing = true;
  //         const _msg = '登陆过期，请重新登录';
  //         const userStore = useAuthStore();
  //         userStore.logout().finally(() => {
  //           message.error(_msg);
  //           isLogoutProcessing = false;
  //         });
  //         // 不再执行下面逻辑
  //         throw new Error(_msg);
  //       }
  //       // default: {
  //       // if (msg) {
  //       //   timeoutMsg = msg;
  //       // }
  //       // }
  //     }
  //     return response;
  //   },
  // });

  return client;
}

export const requestClient = createRequestClient(apiURL, {
  responseReturn: 'data',
});

export const baseRequestClient = new RequestClient({ baseURL: apiURL });
