import type { BaseModel, PageQuery } from '#/api/baseModel';

import { requestClient } from '#/api/request';

export namespace SysRoleApi {
  export interface Role extends BaseModel {
    availableStatus?: number;
    code?: string;
    menuId?: number;
    dataScope?: number;
    // shopDataScope?: number;
    roleKey?: string;
    name?: string;
    remark?: string;
    menuIds?: number[];
    deptIds?: number[];
  }

  // export interface RoleDetailSaveParam {
  //   sysRole: Role;
  //   menuIds?: number[];
  // }
}
/**
 * 查询角色列表
 * @param params
 * @returns {*} page
 */
export function getRoleList(params: PageQuery) {
  return requestClient.get('/system/role/findList', { params });
}
/**
 * 获取所有角色
 * @param params
 * @returns {*} page
 */
export function getAllRole() {
  return requestClient.get('/system/role/getAllRoles');
}

// export function getRoleMallIds(id: number) {
//   return requestClient.get(`/system/role/getRoleMallIds/${id}`);
// }
/**
 * 查询角色详细
 * @param id
 */
export function getRoleDetail(id: number) {
  return requestClient.get(`/system/role/${id}`);
}
/**
 * 修改角色数据权限
 * @param id
 */
export function authDataScope(data: SysRoleApi.Role) {
  return requestClient.post('/system/role/authDataScope', data);
}
/**
 * 获取角色部门id集合
 * @param id 角色id
 * @return 角色部门id集合
 */
export function getRoleDeptIds(id: number) {
  return requestClient.get(`/system/role/getRoleDeptIds/${id}`);
}

// export function authShopDataScope(data: SysRoleApi.Role) {
//   return requestClient.post('/system/role/authShopDataScope', {
//     role: data,
//   });
// }

// export function saveRole(data: SysRoleApi.RoleDetailSaveParam) {
//   return requestClient.post('/system/role/saveRole', data);
// }
/**
 * 新增角色
 * @param data
 */
export function apiAdd(data: SysRoleApi.Role) {
  return requestClient.post('/system/role', data);
}
/**
 * 修改角色
 * @param data
 */
export function apiUpdate(data: SysRoleApi.Role) {
  return requestClient.put('/system/role', data);
}

export function deleteRole(id: number) {
  return requestClient.delete(`/system/role/${id}`);
}
/**
 * 修改角色状态
 * @param id
 */
export function changeRoleStatus(data: SysRoleApi.Role) {
  return requestClient.post('/system/role/changeRoleStatus', data);
}

/**
 * 已分配角色的用户分页
 * @param params 请求参数
 * @returns 分页
 */
export function roleAllocatedList(params?: PageQuery) {
  return requestClient.get('/system/role/authUser/allocatedList', { params });
}

/**
 * 未授权的用户
 * @param params
 * @returns void
 */
export function roleUnallocatedList(params?: PageQuery) {
  return requestClient.get('/system/role/authUser/unallocatedList', {
    params,
  });
}

/**
 * 取消用户角色授权
 * @returns void
 */
export function roleAuthCancel(data: { roleId: number; userId: number }) {
  return requestClient.put('/system/role/authUser/cancel', data, {
    successMessageMode: 'message',
  });
}

/**
 * 批量取消授权
 * @param roleId 角色ID
 * @param userIds 用户ID集合
 * @returns void
 */
export function roleAuthCancelAll(roleId: number, userIds: number[]) {
  return requestClient.put(
    `/system/role/authUser/cancelAll?roleId=${roleId}&userIds=${userIds.join(',')}`,
    undefined,
    { successMessageMode: 'message' },
  );
}

/**
 * 批量授权用户
 * @param roleId 角色ID
 * @param userIds 用户ID集合
 * @returns void
 */
export function roleSelectAll(roleId: number, userIds: number[]) {
  return requestClient.put(
    `/system/role/authUser/selectAll?roleId=${roleId}&userIds=${userIds.join(',')}`,
    undefined,
    { successMessageMode: 'message' },
  );
}
