import type { BaseModel, PageQuery } from '#/api/baseModel';

import { commonExport } from '#/api/helper';
import { requestClient } from '#/api/request';

export namespace TrainingApi {
  export interface Training extends BaseModel {
    /**
     * 员工信息ID
     */
    employeeId?: number;

    /**
     * 姓名
     */
    name?: string;

    /**
     * 身份证号
     */
    idCardNumber?: string;
    /**
     * 培训课程
     */
    trainingCourse?: string;
    /**
     * 培训机构
     */
    trainingInstitution?: string;
    /**
     * 培训讲师
     */
    trainer?: string;
    /**
     * 参训时间
     */
    trainingTime?: string;
    /**
     * 服务期
     */
    serviceEndDate?: string;
    /**
     * 服务年限
     */
    serviceYears?: string;
    /**
     * 培训金额
     */
    trainingAmount?: number;

    /**
     * 备注
     */
    remark?: string;
  }
}

/**
 * 查询员工培训档案列表
 * @param params
 * @returns {*} page
 */
export function apiPage(params: PageQuery) {
  return requestClient.get('/employee/training/page', { params });
}
/**
 * 查询员工培训档案详细
 * @param id
 */
export function apiDetail(id: number) {
  return requestClient.get(`/employee/training/${id}`);
}
/**
 * 新增员工培训档案
 * @param data
 */
export function apiAdd(data: TrainingApi.Training) {
  return requestClient.post('/employee/training', data);
}
/**
 * 修改员工培训档案
 * @param data
 */
export function apiUpdate(data: TrainingApi.Training) {
  return requestClient.put('/employee/training', data);
}
/**
 * 删除员工培训档案
 * @param id
 */
export function apiDelete(id: Array<number> | number) {
  return requestClient.delete(`/employee/training/${id}`);
}
/**
 * 导出员工培训档案
 * @param params
 */
export function apiExport(params: PageQuery) {
  return commonExport('/employee/training/export', params);
}

/**
 * 下载用户导入模板
 * @returns blob
 */
export function downloadImportTemplate() {
  return commonExport('/employee/training/importTemplate', {});
}

/**
 * 从excel导入培训信息
 * @param file
 * @returns void
 */
export function trainingImportData(file: Blob) {
  return requestClient.post<{ errMessage: string; success: boolean }>(
    '/employee/training/import',
    { file },
    {
      headers: {
        'Content-Type': 'multipart/form-data;charset=UTF-8',
      },
      isTransformResponse: false,
      responseReturn: 'body',
      timeout: 300_000,
    },
  );
}
