import type { EmployeeInfoApi } from './employeeModel';

import type { PageQuery } from '#/api/baseModel';

import { commonExport } from '#/api/helper';
import { requestClient } from '#/api/request';

/**
 * 查询员工信息列表
 * @param params
 * @returns {*} page
 */
export function apiPage(params: PageQuery) {
  return requestClient.get('/employee/info/page', { params });
}
/**
 * 查询员工信息详细
 * @param id
 */
export function apiDetail(id: number) {
  return requestClient.get(`/employee/info/${id}`);
}
/**
 * 展示员工信息详细
 * @param id
 */
export function apiInfoDetail(id: number) {
  return requestClient.get(`/employee/info/detail/${id}`);
}
/**
 * 新增员工信息
 * @param data
 */
export function apiAdd(data: EmployeeInfoApi.Employee) {
  return requestClient.post('/employee/info', data);
}
/**
 * 修改员工信息
 * @param data
 */
export function apiUpdate(data: EmployeeInfoApi.Employee) {
  return requestClient.put('/employee/info', data);
}
/**
 * 删除员工信息
 * @param id
 */
export function apiDelete(id: Array<number> | number) {
  return requestClient.delete(`/employee/info/${id}`);
}
/**
 * 导出员工信息列表
 * @param params
 */
export function exportEmployeeList(params: PageQuery) {
  return commonExport('/employee/info/export', params);
}
/**
 * 导出员工信息
 * @param params
 */
export function apiExport(id: number) {
  return commonExport(`/employee/info/export/${id}`, {});
}

/**
 * 申请员工入职
 * @param data
 */
export function applyEntry(data: EmployeeInfoApi.EmployeeApplyBo) {
  return requestClient.post('/employee/info/applyEntry', data);
}
/**
 * 申请员工离职
 * @param data
 */
export function applyResign(data: EmployeeInfoApi.EmployeeResignApplyBo) {
  return requestClient.post('/employee/info/applyResign', data);
}
/**
 * 申请员工调职
 * @param id
 */
export function applyTransfer(data: EmployeeInfoApi.EmployeeTransferApplyBo) {
  return requestClient.post('/employee/info/applyTransfer', data);
}
/**
 * 申请员工入职
 * @param data
 */
export function applyRegular(data: EmployeeInfoApi.EmployeeRegularApplyBo) {
  return requestClient.post('/employee/info/applyRegular', data);
}
/**
 * 申请员工入职
 * @param data
 */
export function applyRenewalContract(
  data: EmployeeInfoApi.EmployeeRenewalContractApplyBo,
) {
  return requestClient.post('/employee/info/applyRenewalContract', data);
}

/**
 * 人事专用 - 获取部门树
 * @returns 部门树
 */
export function employeeDeptTreeSelect() {
  return requestClient.get('/employee/info/deptTree');
}

/**
 * 下载用户导入模板
 * @returns blob
 */
export function downloadImportTemplate() {
  return commonExport('/employee/info/importTemplate', {});
}

/**
 * 从excel导入用户
 * @param file
 * @returns void
 */
export function employeeImportData(file: Blob) {
  return requestClient.post<{ errMessage: string; success: boolean }>(
    '/employee/info/excel/import',
    { file },
    {
      headers: {
        'Content-Type': 'multipart/form-data;charset=UTF-8',
      },
      isTransformResponse: false,
      responseReturn: 'body',
      timeout: 300_000,
    },
  );
}
