package com.anplus.hr.service.impl;

import cn.hutool.core.date.LocalDateTimeUtil;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.anplus.hr.constant.HrFlowEnum;
import com.anplus.hr.domain.EmployeeInfo;
import com.anplus.hr.mapper.EmployeeInfoMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Period;
import java.util.List;

/**
 * @author 刘斌
 * @date 2025/11/5 17:01
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class EmployeeScheduleService {

    private final EmployeeInfoMapper employeeInfoMapper;

    /**
     * 方式1：使用cron表达式
     * 每天凌晨1点执行
     */
    @Scheduled(cron = "0 0 1 * * ?")
    public void midnightTask() {
        executeSafely("刷新员工工龄以及年龄", this::processRefreshInfo);
    }

    /**
     * 刷新员工工龄以及工龄组
     */
    private void processRefreshInfo() {
        List<EmployeeInfo> employeeInfos = employeeInfoMapper.selectList(new LambdaUpdateWrapper<EmployeeInfo>()
                .eq(EmployeeInfo::getEntryApplyStatus, HrFlowEnum.FINISH.getStatus())
                .ne(EmployeeInfo::getResignationApplyStatus, HrFlowEnum.FINISH.getStatus()));
        for (EmployeeInfo employeeInfo : employeeInfos) {
            Period period = LocalDateTimeUtil.betweenPeriod(employeeInfo.getEntryDate(), LocalDate.from(LocalDateTime.now()));
            int years = period.getYears();
            int months = period.getMonths();
            employeeInfo.setYearsOfService(years > 0 ? years + "年" + months + "个月" : months + "个月");
            // TODO 更新员工工龄组

            precessRefreshAge(employeeInfo);
        }
    }

    private void precessRefreshAge(EmployeeInfo employeeInfo) {
        Period period = LocalDateTimeUtil.betweenPeriod(employeeInfo.getBirthDate(), LocalDate.from(LocalDateTime.now()));
        if (employeeInfo.getAge() != period.getYears()) {
            employeeInfo.setAge(period.getYears());
            // TODO 更新员工年龄组
        }
    }

    /**
     * 安全执行方法，包含异常处理
     */
    private void executeSafely(String taskName, Runnable task) {
        log.info("开始执行{}", taskName);
        long startTime = System.currentTimeMillis();

        try {
            task.run();
            long cost = System.currentTimeMillis() - startTime;
            log.info("{}执行完成，耗时：{}ms", taskName, cost);
        } catch (Exception e) {
            log.error("{}执行失败", taskName, e);
            // 这里可以添加告警逻辑，如发送邮件、短信等
        }
    }
}
