package com.anplus.hr.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.alibaba.cola.dto.SingleResponse;
import com.anplus.hr.domain.params.*;
import com.anplus.hr.domain.vo.EmployeeInfoImportVo;
import com.anplus.hr.domain.vo.EmployeeInfoVo;
import com.anplus.hr.service.EmployeeInfoServ;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.Min;
import jakarta.validation.constraints.NotEmpty;
import org.dromara.core.trans.anno.TransMethodResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import top.binfast.common.core.constant.BusinessType;
import top.binfast.common.core.util.ResponseUtils;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.excel.annotion.ExcelExport;
import top.binfast.common.excel.annotion.ExcelStream;
import top.binfast.common.log.annotation.PinSysLog;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

/**
 * 员工信息
 *
 * @author LiuBin
 * @date 2025-10-28
 */
@Validated
@RestController
@RequestMapping("/employee/info")
public class EmployeeInfoCtrl {

    @Resource
    private EmployeeInfoServ employeeInfoServ;

    /**
     * 查询员工信息列表
     */
    @TransMethodResult
    @SaCheckPermission("employee:info:list")
    @GetMapping("/page")
    public PageResponse<EmployeeInfoVo> pageList(EmployeeInfoListParam param) {
        return employeeInfoServ.queryPageList(param);
    }

    /**
     * 导出员工信息列表
     */
    @ExcelExport(template = "导出简历模板.xlsx", fill = true)
    @SaCheckPermission("employee:info:export")
    @PinSysLog(value = "员工信息", businessType = BusinessType.EXPORT)
    @PostMapping("/export/{id}")
    public List<EmployeeInfoVo> export(@PathVariable Long id) {
        EmployeeInfoVo vo = employeeInfoServ.infoDetail(id);
        vo.setOnDuty("1".equals(vo.getEmployeeType()) ? "在职" : "离职");
        return List.of(vo);
    }

    /**
     * 导入员工信息列表
     */
    @PinSysLog(value = "系统用户", businessType = BusinessType.IMPORT)
    @SaCheckPermission("employee:info:import")
    @PostMapping(value = "/excel/import")
    public Response importWithOptions(@ExcelStream(headRowNumber = 2) Stream<EmployeeInfoImportVo> stream, @RequestPart("file") MultipartFile file) {
        return employeeInfoServ.importEmployeeList(stream, file);
    }

    /**
     * 获取导入模板
     */
    @ExcelExport(template = "模板.xlsx", fill = true)
    @PostMapping("/importTemplate")
    public List<EmployeeInfoImportVo> importTemplate() {
        return new ArrayList<>();
    }

    /**
     * 获取员工信息详细信息
     *
     * @param id 主键
     */
    @TransMethodResult
    @SaCheckPermission("employee:info:query")
    @GetMapping("/{id}")
    public SingleResponse<EmployeeInfoVo> getDetail(@PathVariable @Min(1)
                                                    Long id) {
        return SingleResponse.of(employeeInfoServ.queryById(id));
    }

    /**
     * 展示员工信息详细信息
     *
     * @param id 主键
     */
    @TransMethodResult
    @SaCheckPermission("employee:info:query")
    @GetMapping("/detail/{id}")
    public SingleResponse<EmployeeInfoVo> infoDetail(@PathVariable @Min(1)
                                                    Long id) {
        return SingleResponse.of(employeeInfoServ.infoDetail(id));
    }

    /**
     * 新增员工信息
     */
    @SaCheckPermission("employee:info:add")
    @PinSysLog(value = "员工信息", businessType = BusinessType.INSERT)
    @PostMapping()
    public Response add(@Validated(AddGroup.class) @RequestBody EmployeeInfoParam param) {
        return ResponseUtils.ofResult(employeeInfoServ.insertByParam(param));
    }

    /**
     * 修改员工信息
     */
    @SaCheckPermission("employee:info:edit")
    @PinSysLog(value = "员工信息", businessType = BusinessType.UPDATE)
    @PutMapping()
    public Response edit(@Validated(EditGroup.class) @RequestBody EmployeeInfoParam param) {
        return ResponseUtils.ofResult(employeeInfoServ.updateByParam(param));
    }

    /**
     * 申请入职
     */
    @SaCheckPermission("employee:info:add")
    @PinSysLog(value = "员工信息-入职申请", businessType = BusinessType.UPDATE)
    @PostMapping("/applyEntry")
    public Response applyEntry(@RequestBody EmployeeEntryApplyParam param) {
        return ResponseUtils.ofResult(employeeInfoServ.applyEntry(param));
    }

    /**
     * 申请调职
     */
    @SaCheckPermission("employee:info:edit")
    @PinSysLog(value = "员工信息-调职申请", businessType = BusinessType.UPDATE)
    @PostMapping("/applyTransfer")
    public Response applyTransfer(@Validated @RequestBody EmployeeTransferApplyParam param) {
        return ResponseUtils.ofResult(employeeInfoServ.applyTransfer(param));
    }

    /**
     * 申请离职
     */
    @SaCheckPermission("employee:info:resign")
    @PinSysLog(value = "员工信息-离职申请", businessType = BusinessType.UPDATE)
    @PostMapping("/applyResign")
    public Response applyResign(@Validated @RequestBody EmployeeResignApplyParam param) {
        return ResponseUtils.ofResult(employeeInfoServ.applyResign(param));
    }

    /**
     * 删除员工信息
     *
     * @param ids 主键串
     */
    @SaCheckPermission("employee:info:remove")
    @PinSysLog(value = "员工信息", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public Response remove(@NotEmpty(message = "主键不能为空")
                           @PathVariable Long[] ids) {
        return ResponseUtils.ofResult(employeeInfoServ.delByIds(List.of(ids)));
    }
}