package com.anplus.hr.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.anplus.hr.constant.HrFlowEnum;
import com.anplus.hr.constant.HrFlowTypeConstant;
import com.anplus.hr.domain.EmployeeInfo;
import com.anplus.hr.domain.params.*;
import com.anplus.hr.domain.vo.EmployeeInfoImportVo;
import com.anplus.hr.domain.vo.EmployeeInfoVo;
import com.anplus.hr.mapper.EmployeeInfoMapper;
import com.anplus.hr.service.*;
import lombok.RequiredArgsConstructor;
import org.dromara.trans.service.impl.TransService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;
import top.binfast.common.core.enums.ResultCode;
import top.binfast.common.core.util.LambdaUtil;
import top.binfast.common.core.util.MapstructUtils;
import top.binfast.common.excel.core.ExcelContextHolder;
import top.binfast.common.excel.image.CellImageData;
import top.binfast.common.excel.image.ExcelProcessingResult;
import top.binfast.common.excel.image.TempFileExcelImageImporter;
import top.binfast.common.log.model.SysAuditLog;
import top.binfast.common.mybatis.util.QueryUtil;
import top.binfast.common.oss.core.OssService;
import top.binfast.common.oss.entity.UploadResult;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

/**
 * 员工信息Service业务层处理
 *
 * @author LiuBin
 * @date 2025-10-28
 */
@RequiredArgsConstructor
@Service
public class EmployeeInfoServImpl extends ServiceImpl<EmployeeInfoMapper, EmployeeInfo> implements EmployeeInfoServ {

    private final EmployeeInfoMapper employeeInfoMapper;
    private final EmployeeSysDeptServ employeeSysDeptServ;
    private final EmployeeFlowServ employeeFlowServ;
    private final EmployeeAuditLogServ employeeAuditLogServ;
    private final TransService transService;
    private final OssService ossService;

    /**
     * 分页查询员工信息列表
     *
     * @param param 查询条件
     * @return 员工信息分页列表
     */
    @Override
    public PageResponse<EmployeeInfoVo> queryPageList(EmployeeInfoListParam param) {
        Page<EmployeeInfo> page = QueryUtil.getPage(param);
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = this.buildQueryWrapper(param);
        employeeInfoMapper.selectPage(page, lambdaQuery);
        return QueryUtil.getPageResponse(page, MapstructUtils.convert(page.getRecords(), EmployeeInfoVo.class));
    }


    /**
     * 查询符合条件的员工信息列表
     *
     * @param param 查询条件
     * @return 员工信息列表
     */
    @Override
    public List<EmployeeInfoVo> queryList(EmployeeInfoListParam param) {
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = this.buildQueryWrapper(param);
        return MapstructUtils.convert(employeeInfoMapper.selectList(lambdaQuery), EmployeeInfoVo.class);
    }

    private LambdaQueryWrapper<EmployeeInfo> buildQueryWrapper(EmployeeInfoListParam param) {
        Map<String, Object> params = param.getParams();
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = Wrappers.<EmployeeInfo>lambdaQuery();
        lambdaQuery.orderByDesc(EmployeeInfo::getId);
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPlate()), EmployeeInfo::getPlate, param.getPlate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getProject()), EmployeeInfo::getProject, param.getProject());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFirstLevelDepartment()), EmployeeInfo::getFirstLevelDepartment, param.getFirstLevelDepartment());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getSecondLevelDepartment()), EmployeeInfo::getSecondLevelDepartment, param.getSecondLevelDepartment());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmployeeId()), EmployeeInfo::getEmployeeId, param.getEmployeeId());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getJobLevel()), EmployeeInfo::getJobLevel, param.getJobLevel());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPosition()), EmployeeInfo::getPosition, param.getPosition());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPost()), EmployeeInfo::getPost, param.getPost());
        lambdaQuery.like(StrUtil.isNotBlank(param.getName()), EmployeeInfo::getName, param.getName());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getGender()), EmployeeInfo::getGender, param.getGender());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getIdCardNumber()), EmployeeInfo::getIdCardNumber, param.getIdCardNumber());
        lambdaQuery.eq(param.getBirthDate() != null, EmployeeInfo::getBirthDate, param.getBirthDate());
        lambdaQuery.eq(param.getAge() != null, EmployeeInfo::getAge, param.getAge());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getAgeGroup()), EmployeeInfo::getAgeGroup, param.getAgeGroup());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getNativePlace()), EmployeeInfo::getNativePlace, param.getNativePlace());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEthnicity()), EmployeeInfo::getEthnicity, param.getEthnicity());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getMaritalStatus()), EmployeeInfo::getMaritalStatus, param.getMaritalStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPoliticalStatus()), EmployeeInfo::getPoliticalStatus, param.getPoliticalStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPhoneNumber()), EmployeeInfo::getPhoneNumber, param.getPhoneNumber());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmergencyContact()), EmployeeInfo::getEmergencyContact, param.getEmergencyContact());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmergencyContactPhone()), EmployeeInfo::getEmergencyContactPhone, param.getEmergencyContactPhone());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getHomeAddress()), EmployeeInfo::getHomeAddress, param.getHomeAddress());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getHouseholdRegistrationAddress()), EmployeeInfo::getHouseholdRegistrationAddress, param.getHouseholdRegistrationAddress());
        lambdaQuery.eq(param.getWorkStartDate() != null, EmployeeInfo::getWorkStartDate, param.getWorkStartDate());
//        lambdaQuery.eq(param.getEntryDate() != null, EmployeeInfo::getEntryDate, param.getEntryDate());
        lambdaQuery.between(params.get("entryBeginTime") != null && params.get("entryEndTime") != null,
                EmployeeInfo::getEntryDate, params.get("entryBeginTime"), params.get("entryEndTime"));
        lambdaQuery.eq(param.getYearsOfService() != null, EmployeeInfo::getYearsOfService, param.getYearsOfService());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getYearsOfServiceSegment()), EmployeeInfo::getYearsOfServiceSegment, param.getYearsOfServiceSegment());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEducation()), EmployeeInfo::getEducation, param.getEducation());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getDegree()), EmployeeInfo::getDegree, param.getDegree());
        lambdaQuery.eq(param.getGraduationDate() != null, EmployeeInfo::getGraduationDate, param.getGraduationDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getMajor()), EmployeeInfo::getMajor, param.getMajor());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getGraduateSchool()), EmployeeInfo::getGraduateSchool, param.getGraduateSchool());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmployeeType()), EmployeeInfo::getEmployeeType, param.getEmployeeType());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getProfessionalTitle()), EmployeeInfo::getProfessionalTitle, param.getProfessionalTitle());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getResume()), EmployeeInfo::getResume, param.getResume());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmploymentForm()), EmployeeInfo::getEmploymentForm, param.getEmploymentForm());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getContractTerm()), EmployeeInfo::getContractTerm, param.getContractTerm());
        lambdaQuery.eq(param.getContractStartDate() != null, EmployeeInfo::getContractStartDate, param.getContractStartDate());
        lambdaQuery.eq(param.getContractEndDate() != null, EmployeeInfo::getContractEndDate, param.getContractEndDate());
        lambdaQuery.eq(param.getContractExpirationReminder() != null, EmployeeInfo::getContractExpirationReminder, param.getContractExpirationReminder());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getContractSigningStatus()), EmployeeInfo::getContractSigningStatus, param.getContractSigningStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getContractEntity()), EmployeeInfo::getContractEntity, param.getContractEntity());
        lambdaQuery.eq(param.getRegularizationDate() != null, EmployeeInfo::getRegularizationDate, param.getRegularizationDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getTransferStatus()), EmployeeInfo::getTransferStatus, param.getTransferStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getRewardPunishmentStatus()), EmployeeInfo::getRewardPunishmentStatus, param.getRewardPunishmentStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getRemarks()), EmployeeInfo::getRemarks, param.getRemarks());
        lambdaQuery.eq(param.getResignationDate() != null, EmployeeInfo::getResignationDate, param.getResignationDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getResignationReason()), EmployeeInfo::getResignationReason, param.getResignationReason());
        return lambdaQuery;
    }

    /**
     * 导入员工信息列表
     *
     * @param list 员工信息列表
     * @return 是否导入成功
     */
    @Override
    public Response importEmployeeList(Stream<EmployeeInfoImportVo> list, MultipartFile file) {
        List<EmployeeInfoImportVo> errorList = new ArrayList<>();
        List<EmployeeInfoImportVo> successList = new ArrayList<>();
        CompletableFuture<ExcelProcessingResult> future = CompletableFuture.supplyAsync(() -> {
            try {
                return TempFileExcelImageImporter.importExcelWithAllImages(file);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
        list.forEach(item -> {
//            System.out.println(item.getNickName());
//            item.validGroup(AddGroup.class);
//            Set<ConstraintViolation<ExportDemoVo>> validate = ValidatorUtils.validate(item, AddGroup.class);
            if (item.hasError()) {
//                item.addError("业务错误");
                errorList.add(item);
                return;
            }
            EmployeeInfoParam employeeInfoParam = MapstructUtils.convert(item, EmployeeInfoParam.class);
            if (!checkEmployeeIdCardNumberUnique(employeeInfoParam)) {
                item.addError("姓名已存在");
                errorList.add(item);
                return;
            }
            successList.add(item);
        });
        if (CollUtil.isNotEmpty(successList)) {
            Map<String, Long> deptNamesIdMap = employeeSysDeptServ.selectJoinDeptNames();
            ExcelProcessingResult excelProcessingResult = future.join();
            List<EmployeeInfo> insertList = new ArrayList<>(successList.size());
            for (EmployeeInfoImportVo importVo : successList) {
                this.handleImageToUrl(excelProcessingResult, importVo);
                Long leafDeptId = deptNamesIdMap.get(buildDeptNameStr(importVo));
                if (leafDeptId == null) {
                    importVo.addError("部门不存在");
                    errorList.add(importVo);
                    continue;
                }
                EmployeeInfo employeeInfo = MapstructUtils.convert(importVo, EmployeeInfo.class);
                employeeInfo.setDeptId(leafDeptId);
                insertList.add(employeeInfo);
            }
            employeeInfoMapper.insert(insertList);
//            EmployeeDeptCheckAndSaveParam checkAndSaveParam =
//                    new EmployeeDeptCheckAndSaveParam(deptNameGroups, SecurityUtils.getCurrentUserId(), SecurityUtils.getTenantId());
//            employeeDeptServ.checkAndSaveDept(checkAndSaveParam);
        }
        StringBuilder message;
        if (CollUtil.isNotEmpty(errorList)) {
            ExcelContextHolder.setErrorExist();
            message = new StringBuilder("共" + errorList.size() + "条数据导入失败，错误如下：<br/>");
            errorList.forEach(item -> message.append(item.defaultFailMsg()));
            return Response.buildFailure(ResultCode.FAIL.getCode(), message.toString());
        } else {
            message = new StringBuilder("共" + successList.size() + "条数据导入成功");
            Response response = Response.buildSuccess();
            response.setErrMessage(message.toString());
            return response;
        }
    }

    private void handleImageToUrl(ExcelProcessingResult excelProcessingResult, EmployeeInfoImportVo importVo) {
        // 处理图片，保存到OSS，替换成url
        List<CellImageData> imageData = null;
        if (excelProcessingResult.checkWpsOrOffice(importVo.getResumeImage())) {
            imageData = excelProcessingResult.getWpsImageData(importVo.getResumeImage());
        } else {
            imageData = excelProcessingResult.getOfficeImageData(importVo.getLineNum());
        }
        if (CollUtil.isNotEmpty(imageData)) {
            CellImageData cellImageData = imageData.getFirst();
            UploadResult uploadResult = ossService.upload(cellImageData.getImageName(), cellImageData.getImageData());
            importVo.setResumeImage(uploadResult.getUrl());
        }
    }

    private String buildDeptNameStr(EmployeeInfoImportVo importVo) {
        StringBuilder builder = new StringBuilder(importVo.getPlate() + StrUtil.SLASH + importVo.getProject());
        if (StrUtil.isNotBlank(importVo.getFirstLevelDepartment())) {
            builder.append(StrUtil.SLASH).append(importVo.getFirstLevelDepartment());
        }
        if (StrUtil.isNotBlank(importVo.getSecondLevelDepartment())) {
            builder.append(StrUtil.SLASH).append(importVo.getSecondLevelDepartment());
        }
        return builder.toString();
    }

    @Override
    public Boolean checkEmployeeIdCardNumberUnique(EmployeeInfoParam employee) {
        boolean exist = employeeInfoMapper.exists(new LambdaQueryWrapper<EmployeeInfo>()
                .eq(EmployeeInfo::getIdCardNumber, employee.getIdCardNumber())
                .ne(ObjectUtil.isNotNull(employee.getId()), EmployeeInfo::getId, employee.getId()));
        return !exist;
    }

    /**
     * 查询员工信息
     *
     * @param id 主键
     * @return 员工信息
     */
    @Override
    public EmployeeInfoVo queryById(Long id) {
        EmployeeInfo employeeInfo = employeeInfoMapper.selectById(id);
        return MapstructUtils.convert(employeeInfo, EmployeeInfoVo.class);
    }

    /**
     * 获取员工信息详细信息
     *
     * @param id 主键
     * @return 员工信息
     */
    @Override
    public EmployeeInfoVo infoDetail(Long id) {
        EmployeeInfo employeeInfo = employeeInfoMapper.selectById(id);
        EmployeeInfoVo employeeInfoVo = MapstructUtils.convert(employeeInfo, EmployeeInfoVo.class);
        transService.transOne(employeeInfoVo);
        return employeeInfoVo;
    }

    /**
     * 新增员工信息
     *
     * @param param 员工信息
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByParam(EmployeeInfoParam param) {
        EmployeeInfo employeeInfo = MapstructUtils.convert(param, EmployeeInfo.class);
        // 检查或增加部门
        Set<String> deptNameGroups = new HashSet<>();
        deptNameGroups.add(employeeInfo.getFirstLevelDepartment() + StrUtil.DASHED + employeeInfo.getSecondLevelDepartment());
//        EmployeeDeptCheckAndSaveParam checkAndSaveParam =
//                new EmployeeDeptCheckAndSaveParam(deptNameGroups, SecurityUtils.getCurrentUserId(), SecurityUtils.getTenantId());
//        employeeDeptServ.checkAndSaveDept(checkAndSaveParam);
        employeeInfo.setEntryApplyStatus(HrFlowEnum.WAITING.getStatus());
        return this.save(employeeInfo);
    }

    /**
     * 修改员工信息
     *
     * @param param 员工信息
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByParam(EmployeeInfoParam param) {
        EmployeeInfo employeeInfo = MapstructUtils.convert(param, EmployeeInfo.class);
        return this.updateById(employeeInfo);
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(EmployeeInfo entity) {
        // 做一些数据校验,如唯一约束
    }

    /**
     * 员工入职申请
     *
     * @param param 入职申请参数
     * @return 是否申请成功
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean applyEntry(EmployeeEntryApplyParam param) {
        EmployeeFlowParam flowParam = new EmployeeFlowParam();
        flowParam.setFlowCode(HrFlowTypeConstant.ENTRY_CODE);
        flowParam.setFlowType(HrFlowTypeConstant.Entry);
        flowParam.setEmployeeId(param.getId());
        flowParam.setRemark(param.getRemark());
        employeeFlowServ.submitAndFlowStart(flowParam);
        return true;
    }

    /**
     * 员工调岗申请
     *
     * @param param 调岗申请参数
     * @return 是否申请成功
     */
    @Override
    public Boolean applyTransfer(EmployeeTransferApplyParam param) {
        List<SysAuditLog> list = new ArrayList<>(6);
        SysAuditLog firstAuditLog = new SysAuditLog();
        firstAuditLog.setAuditName("调职申请");
        firstAuditLog.setAuditField("firstLevelDepartment");
        firstAuditLog.setAuditFieldName("一级部门");
        firstAuditLog.setAfterVal(param.getFirstLevelDepartment());

        SysAuditLog secondAuditLog = new SysAuditLog();
        secondAuditLog.setAuditName("调职申请");
        secondAuditLog.setAuditField("secondLevelDepartment");
        secondAuditLog.setAuditFieldName("二级部门");
        secondAuditLog.setAfterVal(param.getSecondLevelDepartment());
        list.add(firstAuditLog);
        list.add(secondAuditLog);
        employeeAuditLogServ.saveAuditLogs(list);
        String logIds = LambdaUtil.join(list, (item) -> item.getId() + StrUtil.EMPTY, StrUtil.COMMA);
        EmployeeFlowParam flowParam = new EmployeeFlowParam();
        flowParam.setFlowCode(HrFlowTypeConstant.TRANSFER_CODE);
        flowParam.setFlowType(HrFlowTypeConstant.TRANSFER);
        flowParam.setEmployeeId(param.getId());
        flowParam.setRemark(param.getRemark());
        flowParam.setLogIds(logIds);
        employeeFlowServ.submitAndFlowStart(flowParam);
        return true;
    }

    /**
     * 员工离职申请
     *
     * @param param 入职申请参数
     * @return 是否申请成功
     */
    @Override
    @DSTransactional(rollbackFor = Exception.class)
    public Boolean applyResign(EmployeeResignApplyParam param) {
        List<SysAuditLog> list = new ArrayList<>(6);
        if (StrUtil.isNotBlank(param.getResignReason())) {
            SysAuditLog reasonAuditLog = new SysAuditLog();
            reasonAuditLog.setAuditName("离职申请");
            reasonAuditLog.setAuditField("resignationReason");
            reasonAuditLog.setAuditFieldName("离职原因");
            reasonAuditLog.setAfterVal(param.getResignReason());
            list.add(reasonAuditLog);
        }

        SysAuditLog dateAuditLog = new SysAuditLog();
        dateAuditLog.setAuditName("离职申请");
        dateAuditLog.setAuditField("resignationDate");
        dateAuditLog.setAuditFieldName("离职时间");
        dateAuditLog.setAfterVal(LocalDateTimeUtil.format(param.getResignDate(), DatePattern.NORM_DATE_PATTERN));
        list.add(dateAuditLog);
        employeeAuditLogServ.saveAuditLogs(list);
        String logIds = LambdaUtil.join(list, (item) -> item.getId() + StrUtil.EMPTY, StrUtil.COMMA);
        EmployeeFlowParam flowParam = new EmployeeFlowParam();
        flowParam.setFlowCode(HrFlowTypeConstant.RESIGN_CODE);
        flowParam.setFlowType(HrFlowTypeConstant.RESIGN);
        flowParam.setEmployeeId(param.getId());
        flowParam.setRemark(param.getRemark());
        flowParam.setLogIds(logIds);
        employeeFlowServ.submitAndFlowStart(flowParam);
        return true;
    }

    /**
     * 校验并批量删除员工信息信息
     *
     * @param ids 待删除的主键集合
     * @return 是否删除成功
     */
    @Override
//    @Transactional(rollbackFor = {Exception.class})
    public Boolean delByIds(List<Long> ids) {
        //做一些业务上的校验,判断是否需要校验
        return this.removeByIds(ids);
    }
}
