package com.anplus.hr.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.cola.dto.PageResponse;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.anplus.hr.constant.HrDeptLevel;
import com.anplus.hr.domain.EmployeeDept;
import com.anplus.hr.domain.params.EmployeeDeptCheckAndSaveParam;
import com.anplus.hr.domain.params.EmployeeDeptListParam;
import com.anplus.hr.domain.params.EmployeeDeptParam;
import com.anplus.hr.domain.vo.EmployeeDeptVo;
import com.anplus.hr.mapper.EmployeeDeptMapper;
import com.anplus.hr.service.EmployeeDeptServ;
import lombok.RequiredArgsConstructor;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import top.binfast.common.core.constant.NormalStatus;
import top.binfast.common.core.util.LambdaUtil;
import top.binfast.common.core.util.MapstructUtils;
import top.binfast.common.mybatis.util.QueryUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 员工部门Service业务层处理
 *
 * @author LiuBin
 * @date 2025-10-30
 */
@RequiredArgsConstructor
@Service
public class EmployeeDeptServImpl extends ServiceImpl<EmployeeDeptMapper, EmployeeDept> implements EmployeeDeptServ {

    private final EmployeeDeptMapper employeeDeptMapper;

    /**
     * 分页查询员工部门列表
     *
     * @param param 查询条件
     * @return 员工部门分页列表
     */
    @Override
    public PageResponse<EmployeeDeptVo> queryPageList(EmployeeDeptListParam param) {
        Page<EmployeeDept> page = QueryUtil.getPage(param);
        LambdaQueryWrapper<EmployeeDept> lambdaQuery = this.buildQueryWrapper(param);
        employeeDeptMapper.selectPage(page, lambdaQuery);
        return QueryUtil.getPageResponse(page, MapstructUtils.convert(page.getRecords(), EmployeeDeptVo.class));
    }


    /**
     * 查询符合条件的员工部门列表
     *
     * @param param 查询条件
     * @return 员工部门列表
     */
    @Override
    public List<EmployeeDeptVo> queryList(EmployeeDeptListParam param) {
        LambdaQueryWrapper<EmployeeDept> lambdaQuery = this.buildQueryWrapper(param);
        return MapstructUtils.convert(employeeDeptMapper.selectList(lambdaQuery), EmployeeDeptVo.class);
    }

    private LambdaQueryWrapper<EmployeeDept> buildQueryWrapper(EmployeeDeptListParam param) {
        LambdaQueryWrapper<EmployeeDept> lambdaQuery = Wrappers.<EmployeeDept>lambdaQuery();
        lambdaQuery.orderByDesc(EmployeeDept::getId);
        lambdaQuery.eq(param.getParentId() != null, EmployeeDept::getParentId, param.getParentId());
        lambdaQuery.like(StrUtil.isNotBlank(param.getName()), EmployeeDept::getName, param.getName());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getNodePath()), EmployeeDept::getNodePath, param.getNodePath());
        lambdaQuery.eq(param.getOrderNum() != null, EmployeeDept::getOrderNum, param.getOrderNum());
        lambdaQuery.eq(param.getStatus() != null, EmployeeDept::getStatus, param.getStatus());
        return lambdaQuery;
    }

    /**
     * 批量保存员工部门
     *
     * @param param 部门名称集合
     */
    @Override
    @Async
    @Transactional(rollbackFor = Exception.class)
    public void checkAndSaveDept(EmployeeDeptCheckAndSaveParam param) {
        Set<String> deptNames = param.getDeptNames();
        if (CollUtil.isEmpty(deptNames)) {
            return;
        }
        List<EmployeeDept> updateList = new ArrayList<>();
        for (String deptNameGroup : deptNames) {
            String[] split = deptNameGroup.split(StrUtil.DASHED);
            if (split.length != 2) {
                continue;
            }
            boolean exists = employeeDeptMapper.exists(new LambdaUpdateWrapper<EmployeeDept>()
                    .eq(EmployeeDept::getParentName, split[0])
                    .eq(EmployeeDept::getName, split[1]));
            if (!exists) {
                EmployeeDept parentDept = employeeDeptMapper.selectOne(new LambdaUpdateWrapper<EmployeeDept>()
                        .eq(EmployeeDept::getName, split[0]));
                if (parentDept == null) {
                    parentDept = new EmployeeDept();
                    parentDept.setParentId(0L);
                    parentDept.setParentName("顶级部门");
                    parentDept.setName(split[0]);
                    parentDept.setStatus(NormalStatus.OPEN_STATUS);
                    parentDept.setLevel(HrDeptLevel.DEPT_LEVEL_ONE);
                    parentDept.setTenantId(param.getTenantId());
                    parentDept.setCreateBy(param.getUserId());
                    employeeDeptMapper.insert(parentDept);
                    parentDept.setOrderNum(parentDept.getId());
                    parentDept.setNodePath("/0/" + parentDept.getId());
                    updateList.add(parentDept);
                }
                EmployeeDept employeeDept = new EmployeeDept();
                employeeDept.setParentId(parentDept.getId());
                employeeDept.setParentName(parentDept.getName());
                employeeDept.setName(split[1]);
                employeeDept.setStatus(NormalStatus.OPEN_STATUS);
                parentDept.setLevel(HrDeptLevel.DEPT_LEVEL_TWO);
                employeeDept.setTenantId(param.getTenantId());
                employeeDept.setCreateBy(param.getUserId());
                employeeDeptMapper.insert(employeeDept);
                employeeDept.setOrderNum(employeeDept.getId());
                employeeDept.setNodePath(parentDept.getNodePath() + "/" + employeeDept.getId() + "/");
                updateList.add(employeeDept);
            }
        }
        if (CollUtil.isNotEmpty(updateList)) {
            employeeDeptMapper.updateById(updateList);
        }
    }

    /**
     * 根据部门等级查询部门名称
     *
     * @param level 部门等级
     * @return 部门名称列表
     */
    @Override
    public List<String> selectDeptNamesByLevel(Integer level) {
        List<EmployeeDept> employeeDeptList = employeeDeptMapper.selectList(new LambdaUpdateWrapper<EmployeeDept>()
                .eq(EmployeeDept::getLevel, level));
        return LambdaUtil.mapToList(employeeDeptList, EmployeeDept::getName);
    }

    @Override
    public List<String> selectDeptNamesByParentName(String parentName) {
        List<EmployeeDept> employeeDeptList = employeeDeptMapper.selectList(new LambdaUpdateWrapper<EmployeeDept>()
                .eq(EmployeeDept::getParentName, parentName));
        return LambdaUtil.mapToList(employeeDeptList, EmployeeDept::getName);
    }

    /**
     * 查询员工部门
     *
     * @param id 主键
     * @return 员工部门
     */
    @Override
    public EmployeeDeptVo queryById(Long id) {
        EmployeeDept employeeDept = employeeDeptMapper.selectById(id);
        return MapstructUtils.convert(employeeDept, EmployeeDeptVo.class);
    }

    /**
     * 新增员工部门
     *
     * @param param 员工部门
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByParam(EmployeeDeptParam param) {
        EmployeeDept employeeDept = MapstructUtils.convert(param, EmployeeDept.class);
        return this.save(employeeDept);
    }

    /**
     * 修改员工部门
     *
     * @param param 员工部门
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByParam(EmployeeDeptParam param) {
        EmployeeDept employeeDept = MapstructUtils.convert(param, EmployeeDept.class);
        return this.updateById(employeeDept);
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(EmployeeDept entity) {
        // 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除员工部门信息
     *
     * @param ids 待删除的主键集合
     * @return 是否删除成功
     */
    @Override
//    @Transactional(rollbackFor = {Exception.class})
    public Boolean delByIds(List<Long> ids) {
        //做一些业务上的校验,判断是否需要校验
        return this.removeByIds(ids);
    }
}
