package com.anplus.hr.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import com.alibaba.cola.dto.MultiResponse;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.alibaba.cola.dto.SingleResponse;
import com.anplus.hr.domain.params.EmployeeDeptListParam;
import com.anplus.hr.domain.params.EmployeeDeptParam;
import com.anplus.hr.domain.vo.EmployeeDeptVo;
import com.anplus.hr.service.EmployeeDeptServ;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.Min;
import jakarta.validation.constraints.NotBlank;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import top.binfast.common.core.constant.BusinessType;
import top.binfast.common.core.util.ResponseUtils;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.excel.annotion.ExcelExport;
import top.binfast.common.log.annotation.PinSysLog;

import java.util.List;

/**
 * 员工部门
 *
 * @author LiuBin
 * @date 2025-10-30
 */
@Validated
@RestController
@RequestMapping("/employee/dept")
public class EmployeeDeptCtrl {

    @Resource
    private EmployeeDeptServ employeeDeptServ;

    /**
     * 查询员工部门列表
     */
    @SaCheckPermission("employee:dept:list")
    @GetMapping("/page")
    public PageResponse<EmployeeDeptVo> pageList(EmployeeDeptListParam param) {
        return employeeDeptServ.queryPageList(param);
    }

    /**
     * 导出员工部门列表
     */
    @ExcelExport
    @SaCheckPermission("employee:dept:export")
    @PinSysLog(value = "员工部门", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public List<EmployeeDeptVo> export(EmployeeDeptListParam param) {
        return employeeDeptServ.queryList(param);
    }

    /**
     * 根据部门级别查询部门名称
     *
     * @param level 部门级别
     */
    @GetMapping("/deptNames/{level}")
    public MultiResponse<String> getDeptNamesByLevel(@PathVariable @Min(1) Integer level) {
        return MultiResponse.of(employeeDeptServ.selectDeptNamesByLevel(level));
    }

    /**
     * 根据父级部门名称查询部门名称
     *
     * @param parentName 父级部门名称
     */
    @GetMapping("/deptNamesByParentName/{parentName}")
    public MultiResponse<String> getDeptNamesByParentName(@PathVariable @NotBlank String parentName) {
        return MultiResponse.of(employeeDeptServ.selectDeptNamesByParentName(parentName));
    }

    /**
     * 获取员工部门详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("employee:dept:query")
    @GetMapping("/{id}")
    public SingleResponse<EmployeeDeptVo> getDetail(@PathVariable @Min(1)
                                                    Long id) {
        return SingleResponse.of(employeeDeptServ.queryById(id));
    }

    /**
     * 新增员工部门
     */
    @SaCheckPermission("employee:dept:add")
    @PinSysLog(value = "员工部门", businessType = BusinessType.INSERT)
    @PostMapping()
    public Response add(@Validated(AddGroup.class) @RequestBody EmployeeDeptParam param) {
        return ResponseUtils.ofResult(employeeDeptServ.insertByParam(param));
    }

    /**
     * 修改员工部门
     */
    @SaCheckPermission("employee:dept:edit")
    @PinSysLog(value = "员工部门", businessType = BusinessType.UPDATE)
    @PutMapping()
    public Response edit(@Validated(EditGroup.class) @RequestBody EmployeeDeptParam param) {
        return ResponseUtils.ofResult(employeeDeptServ.updateByParam(param));
    }

    /**
     * 删除员工部门
     *
     * @param ids 主键串
     */
    @SaCheckPermission("employee:dept:remove")
    @PinSysLog(value = "员工部门", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public Response remove(@NotEmpty(message = "主键不能为空")
                           @PathVariable Long[] ids) {
        return ResponseUtils.ofResult(employeeDeptServ.delByIds(List.of(ids)));
    }
}