package com.anplus.hr.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckPermission;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.alibaba.cola.dto.SingleResponse;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.Min;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import top.binfast.common.core.constant.BusinessType;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.core.util.ResponseUtils;
import top.binfast.common.excel.annotion.ExcelExport;
import top.binfast.common.log.annotation.PinSysLog;
import com.anplus.hr.domain.params.EmployeeLaborCostListParam;
import com.anplus.hr.domain.params.EmployeeLaborCostParam;
import com.anplus.hr.domain.vo.EmployeeLaborCostVo;
import com.anplus.hr.service.EmployeeLaborCostServ;

/**
 * 人工成本档案
 *
 * @author LiuBin
 * @date 2025-12-09
 */
@Validated
@RestController
@RequestMapping("/employee/laborCost")
public class EmployeeLaborCostCtrl {

    @Resource
    private EmployeeLaborCostServ employeeLaborCostServ;

    /**
     * 查询人工成本档案列表
     */
    @SaCheckPermission("employee:laborCost:list")
    @GetMapping("/page")
    public PageResponse<EmployeeLaborCostVo> pageList(EmployeeLaborCostListParam param) {
        return employeeLaborCostServ.queryPageList(param);
    }

    /**
     * 导出人工成本档案列表
     */
    @ExcelExport
    @SaCheckPermission("employee:laborCost:export")
    @PinSysLog(value = "人工成本档案", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public List<EmployeeLaborCostVo> export(EmployeeLaborCostListParam param) {
        return employeeLaborCostServ.queryList(param);
    }

    /**
     * 获取人工成本档案详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("employee:laborCost:query")
    @GetMapping("/{id}")
    public SingleResponse<EmployeeLaborCostVo> getDetail(@PathVariable @Min(1)
                                                    Long id) {
        return SingleResponse.of(employeeLaborCostServ.queryById(id));
    }

    /**
     * 新增人工成本档案
     */
    @SaCheckPermission("employee:laborCost:add")
    @PinSysLog(value = "人工成本档案", businessType = BusinessType.INSERT)
    @PostMapping()
    public Response add(@Validated(AddGroup.class) @RequestBody EmployeeLaborCostParam param) {
        return ResponseUtils.ofResult(employeeLaborCostServ.insertByParam(param));
    }

    /**
     * 修改人工成本档案
     */
    @SaCheckPermission("employee:laborCost:edit")
    @PinSysLog(value = "人工成本档案", businessType = BusinessType.UPDATE)
    @PutMapping()
    public Response edit(@Validated(EditGroup.class) @RequestBody EmployeeLaborCostParam param) {
        return ResponseUtils.ofResult(employeeLaborCostServ.updateByParam(param));
    }

    /**
     * 删除人工成本档案
     *
     * @param ids 主键串
     */
    @SaCheckPermission("employee:laborCost:remove")
    @PinSysLog(value = "人工成本档案", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public Response remove(@NotEmpty(message = "主键不能为空")
                           @PathVariable Long[] ids) {
        return ResponseUtils.ofResult(employeeLaborCostServ.delByIds(List.of(ids)));
    }
}