package com.anplus.hr.service.impl;

import cn.hutool.core.util.StrUtil;
import com.anplus.hr.domain.EmployeeInfo;
import com.anplus.hr.domain.params.EmployeeChangeAnalysisListParam;
import com.anplus.hr.domain.vo.EmployeeChangeAnalysisVo;
import com.anplus.hr.mapper.EmployeeInfoMapper;
import com.anplus.hr.service.EmployeeChangeAnalysisServ;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;
import top.binfast.common.core.exception.PlatformException;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;

/**
 * 人员变化分析Service业务层处理
 *
 * @author LiuBin
 * @date 2025-11-23
 */
@RequiredArgsConstructor
@Service
public class EmployeeChangeAnalysisServImpl extends ServiceImpl<EmployeeInfoMapper, EmployeeInfo> implements EmployeeChangeAnalysisServ {

    private final EmployeeInfoMapper employeeInfoMapper;

    /**
     * 分页查询人员变化分析列表
     *
     * @param param 查询条件
     * @return 人员变化分析分页列表
     */
//    @Override
//    public PageResponse<EmployeeChangeAnalysisVo> queryPageList(EmployeeChangeAnalysisListParam param) {
//        Page<EmployeeChangeAnalysis> page = QueryUtil.getPage(param);
//        LambdaQueryWrapper<EmployeeChangeAnalysis> lambdaQuery = this.buildQueryWrapper(param);
//        employeeChangeAnalysisMapper.selectPage(page, lambdaQuery);
//        return QueryUtil.getPageResponse(page, MapstructUtils.convert(page.getRecords(), EmployeeChangeAnalysisVo.class));
//    }


    /**
     * 查询符合条件的人员变化分析列表
     *
     * @param param 查询条件
     * @return 人员变化分析列表
     */
    @Override
    public List<EmployeeChangeAnalysisVo> queryList(EmployeeChangeAnalysisListParam param) {
//        LambdaQueryWrapper<EmployeeChangeAnalysis> lambdaQuery = this.buildQueryWrapper(param);
//        Map<String, Object> params = param.getParams();
        if (StrUtil.isBlank(param.getBeginTime()) || StrUtil.isBlank(param.getEndTime())) {
            throw new PlatformException("请选择时间范围");
        }
        List<EmployeeChangeAnalysisVo> analysisVoList = employeeInfoMapper.queryChangeAnalysisList(param.getBeginTime(), param.getEndTime());
        if (!analysisVoList.isEmpty()) {
            EmployeeChangeAnalysisVo sumVo = getChangeAnalysisSumVo(analysisVoList);
            analysisVoList.add(sumVo);
            for (EmployeeChangeAnalysisVo analysisVo : analysisVoList) {
                analysisVo.setProbationResignationRate(calcResignationRate(analysisVo.getProbationPeriod(), analysisVo.getEndingCount()));
                analysisVo.setWithin3YearsResignationRate(calcResignationRate(analysisVo.getWithin3Years(), analysisVo.getEndingCount()));
                analysisVo.setOver3YearsResignationRate(calcResignationRate(analysisVo.getOver3Years(), analysisVo.getEndingCount()));
                analysisVo.setPassiveResignationRate(calcResignationRate(analysisVo.getPassiveResignation(), analysisVo.getEndingCount()));
                analysisVo.setTotalResignationRate(calcResignationRate(analysisVo.getTotalResignation(), analysisVo.getEndingCount()));
            }
        }
        return analysisVoList;
    }

    @NotNull
    private static EmployeeChangeAnalysisVo getChangeAnalysisSumVo(List<EmployeeChangeAnalysisVo> analysisVoList) {
        EmployeeChangeAnalysisVo sumVo = new EmployeeChangeAnalysisVo("合计");
        for (EmployeeChangeAnalysisVo changeAnalysisVo : analysisVoList) {
            sumVo.setBeginningCount(sumVo.getBeginningCount() + changeAnalysisVo.getBeginningCount());
            sumVo.setEntryCount(sumVo.getEntryCount() + changeAnalysisVo.getEntryCount());
            sumVo.setRegularCount(sumVo.getRegularCount() + changeAnalysisVo.getRegularCount());
            sumVo.setTransferIn(sumVo.getTransferIn() + changeAnalysisVo.getTransferIn());
            sumVo.setTransferOut(sumVo.getTransferOut() + changeAnalysisVo.getTransferOut());
            sumVo.setProbationPeriod(sumVo.getProbationPeriod() + changeAnalysisVo.getProbationPeriod());
            sumVo.setWithin3Years(sumVo.getWithin3Years() + changeAnalysisVo.getWithin3Years());
            sumVo.setOver3Years(sumVo.getOver3Years() + changeAnalysisVo.getOver3Years());
            sumVo.setPassiveResignation(sumVo.getPassiveResignation() + changeAnalysisVo.getPassiveResignation());
            sumVo.setTotalResignation(sumVo.getTotalResignation() + changeAnalysisVo.getTotalResignation());
            sumVo.setEndingCount(sumVo.getEndingCount() + changeAnalysisVo.getEndingCount());
        }
        return sumVo;
    }

    private String calcResignationRate(Integer count, Integer total) {
        if (count == null || count == 0 || total == null || total == 0) {
            return "0.00%";
        }
        String rate = new BigDecimal(count).divide(new BigDecimal(total + count), 4, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(2, RoundingMode.HALF_UP).toString();
        return rate + "%";
    }

//    private LambdaQueryWrapper<EmployeeChangeAnalysis> buildQueryWrapper(EmployeeChangeAnalysisListParam param) {
//        LambdaQueryWrapper<EmployeeChangeAnalysis> lambdaQuery = Wrappers.<EmployeeChangeAnalysis>lambdaQuery();
//        lambdaQuery.orderByDesc(EmployeeChangeAnalysis::getId);
//        lambdaQuery.eq(StrUtil.isNotBlank(param.getPlate()), EmployeeChangeAnalysis::getPlate, param.getPlate());
//        lambdaQuery.eq(StrUtil.isNotBlank(param.getPlateId()), EmployeeChangeAnalysis::getPlateId, param.getPlateId());
//        lambdaQuery.eq(param.getBeginningCount() != null, EmployeeChangeAnalysis::getBeginningCount, param.getBeginningCount());
//        lambdaQuery.eq(param.getEntryCount() != null, EmployeeChangeAnalysis::getEntryCount, param.getEntryCount());
//        lambdaQuery.eq(param.getRegularCount() != null, EmployeeChangeAnalysis::getRegularCount, param.getRegularCount());
//        lambdaQuery.eq(param.getTransferIn() != null, EmployeeChangeAnalysis::getTransferIn, param.getTransferIn());
//        lambdaQuery.eq(param.getTransferOut() != null, EmployeeChangeAnalysis::getTransferOut, param.getTransferOut());
//        lambdaQuery.eq(param.getProbationPeriod() != null, EmployeeChangeAnalysis::getProbationPeriod, param.getProbationPeriod());
//        lambdaQuery.eq(param.getWithin3Years() != null, EmployeeChangeAnalysis::getWithin3Years, param.getWithin3Years());
//        lambdaQuery.eq(param.getOver3Years() != null, EmployeeChangeAnalysis::getOver3Years, param.getOver3Years());
//        lambdaQuery.eq(param.getPassiveResignation() != null, EmployeeChangeAnalysis::getPassiveResignation, param.getPassiveResignation());
//        lambdaQuery.eq(param.getTotalResignation() != null, EmployeeChangeAnalysis::getTotalResignation, param.getTotalResignation());
//        lambdaQuery.eq(param.getEndingCount() != null, EmployeeChangeAnalysis::getEndingCount, param.getEndingCount());
//        lambdaQuery.eq(param.getProbationResignationRate() != null, EmployeeChangeAnalysis::getProbationResignationRate, param.getProbationResignationRate());
//        lambdaQuery.eq(param.getWithin3YearsResignationRate() != null, EmployeeChangeAnalysis::getWithin3YearsResignationRate, param.getWithin3YearsResignationRate());
//        lambdaQuery.eq(param.getOver3YearsResignationRate() != null, EmployeeChangeAnalysis::getOver3YearsResignationRate, param.getOver3YearsResignationRate());
//        lambdaQuery.eq(param.getPassiveResignationRate() != null, EmployeeChangeAnalysis::getPassiveResignationRate, param.getPassiveResignationRate());
//        lambdaQuery.eq(param.getTotalResignationRate() != null, EmployeeChangeAnalysis::getTotalResignationRate, param.getTotalResignationRate());
//        return lambdaQuery;
//    }

    /**
     * 查询人员变化分析
     *
     * @param id 主键
     * @return 人员变化分析
     */
//    @Override
//    public EmployeeChangeAnalysisVo queryById(Long id){
//        EmployeeChangeAnalysis employeeChangeAnalysis = employeeChangeAnalysisMapper.selectById(id);
//        return MapstructUtils.convert(employeeChangeAnalysis, EmployeeChangeAnalysisVo.class);
//    }
//
//    /**
//     * 新增人员变化分析
//     *
//     * @param param 人员变化分析
//     * @return 是否新增成功
//     */
//    @Override
//    public Boolean insertByParam(EmployeeChangeAnalysisParam param) {
//        EmployeeChangeAnalysis employeeChangeAnalysis = MapstructUtils.convert(param, EmployeeChangeAnalysis.class);
//        return this.save(employeeChangeAnalysis);
//    }
//
//    /**
//     * 修改人员变化分析
//     *
//     * @param param 人员变化分析
//     * @return 是否修改成功
//     */
//    @Override
//    public Boolean updateByParam(EmployeeChangeAnalysisParam param) {
//        EmployeeChangeAnalysis employeeChangeAnalysis = MapstructUtils.convert(param, EmployeeChangeAnalysis.class);
//        return this.updateById(employeeChangeAnalysis);
//    }
//
//    /**
//     * 保存前的数据校验
//     */
//    private void validEntityBeforeSave(EmployeeChangeAnalysis entity){
//        // 做一些数据校验,如唯一约束
//    }
//
//    /**
//     * 校验并批量删除人员变化分析信息
//     *
//     * @param ids 待删除的主键集合
//     * @return 是否删除成功
//     */
//    @Override
//    @Transactional(rollbackFor = {Exception.class})
//    public Boolean delByIds(List<Long> ids) {
//        //做一些业务上的校验,判断是否需要校验
//        return this.removeByIds(ids);
//    }
}
