package com.anplus.hr.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.alibaba.cola.dto.SingleResponse;
import com.anplus.hr.domain.params.EmployeeFlowListParam;
import com.anplus.hr.domain.params.EmployeeFlowParam;
import com.anplus.hr.domain.vo.EmployeeFlowVo;
import com.anplus.hr.service.EmployeeFlowServ;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.Min;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import top.binfast.common.core.constant.BusinessType;
import top.binfast.common.core.util.ResponseUtils;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.excel.annotion.ExcelExport;
import top.binfast.common.idempotent.annotation.NoRepeatSubmit;
import top.binfast.common.log.annotation.PinSysLog;

import java.util.List;

/**
 * 人事审批对象
 *
 * @author LiuBin
 * @date 2025-10-31
 */
@Validated
@RestController
@RequestMapping("/employee/flow")
public class EmployeeFlowCtrl {

    @Resource
    private EmployeeFlowServ employeeFlowServ;

    /**
     * 查询人事审批对象列表
     */
    @SaCheckPermission("employee:flow:list")
    @GetMapping("/page")
    public PageResponse<EmployeeFlowVo> pageList(EmployeeFlowListParam param) {
        return employeeFlowServ.queryPageList(param);
    }

    /**
     * 导出人事审批对象列表
     */
    @ExcelExport
    @SaCheckPermission("employee:flow:export")
    @PinSysLog(value = "人事审批对象", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public List<EmployeeFlowVo> export(EmployeeFlowListParam param) {
        return employeeFlowServ.queryList(param);
    }

    /**
     * 获取人事审批对象详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("employee:flow:query")
    @GetMapping("/{id}")
    public SingleResponse<EmployeeFlowVo> getDetail(@PathVariable @Min(1)
                                                    Long id) {
        return SingleResponse.of(employeeFlowServ.queryById(id));
    }

    /**
     * 新增人事审批对象
     */
    @SaCheckPermission("employee:flow:add")
    @PinSysLog(value = "人事审批对象", businessType = BusinessType.INSERT)
    @PostMapping()
    public SingleResponse<EmployeeFlowVo> add(@Validated(AddGroup.class) @RequestBody EmployeeFlowParam param) {
        return SingleResponse.of(employeeFlowServ.insertByParam(param));
    }

    /**
     * 修改人事审批对象
     */
    @SaCheckPermission("employee:flow:edit")
    @PinSysLog(value = "人事审批对象", businessType = BusinessType.UPDATE)
    @PutMapping()
    public SingleResponse<EmployeeFlowVo> edit(@Validated(EditGroup.class) @RequestBody EmployeeFlowParam param) {
        return SingleResponse.of(employeeFlowServ.updateByParam(param));
    }

    /**
     * 提交请假并提交流程
     */
    @SaCheckPermission("employee:leave:add")
    @PinSysLog(value = "发起人事审批", businessType = BusinessType.INSERT)
    @NoRepeatSubmit()
    @PostMapping("/submitAndFlowStart")
    public SingleResponse<EmployeeFlowVo> submitAndFlowStart(@Validated(AddGroup.class) @RequestBody EmployeeFlowParam param) {
        return SingleResponse.of(employeeFlowServ.submitAndFlowStart(param));
    }

    /**
     * 删除人事审批对象
     *
     * @param ids 主键串
     */
    @SaCheckPermission("employee:flow:remove")
    @PinSysLog(value = "人事审批对象", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public Response remove(@NotEmpty(message = "主键不能为空")
                           @PathVariable Long[] ids) {
        return ResponseUtils.ofResult(employeeFlowServ.delByIds(List.of(ids)));
    }
}