package com.anplus.hr.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.anplus.hr.domain.EmployeeInfo;
import com.anplus.hr.domain.InterviewProfile;
import com.anplus.hr.domain.params.InterviewProfileListParam;
import com.anplus.hr.domain.params.InterviewProfileParam;
import com.anplus.hr.domain.vo.InterviewProfileImportVo;
import com.anplus.hr.domain.vo.InterviewProfileVo;
import com.anplus.hr.mapper.EmployeeInfoMapper;
import com.anplus.hr.mapper.InterviewProfileMapper;
import com.anplus.hr.service.InterviewProfileServ;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import top.binfast.common.core.enums.ResultCode;
import top.binfast.common.core.util.LambdaUtil;
import top.binfast.common.core.util.MapstructUtils;
import top.binfast.common.mybatis.bean.model.BaseModel;
import top.binfast.common.mybatis.util.QueryUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

/**
 * 面试档案Service业务层处理
 *
 * @author LiuBin
 * @date 2025-12-08
 */
@RequiredArgsConstructor
@Service
public class InterviewProfileServImpl extends ServiceImpl<InterviewProfileMapper, InterviewProfile> implements InterviewProfileServ {

    private final InterviewProfileMapper interviewProfileMapper;
    private final EmployeeInfoMapper employeeInfoMapper;

    /**
     * 分页查询面试档案列表
     *
     * @param param 查询条件
     * @return 面试档案分页列表
     */
    @Override
    public PageResponse<InterviewProfileVo> queryPageList(InterviewProfileListParam param) {
        Page<InterviewProfile> page = QueryUtil.getPage(param);
        LambdaQueryWrapper<InterviewProfile> lambdaQuery = this.buildQueryWrapper(param);
        interviewProfileMapper.selectPage(page, lambdaQuery);
        return QueryUtil.getPageResponse(page, MapstructUtils.convert(page.getRecords(), InterviewProfileVo.class));
    }


    /**
     * 查询符合条件的面试档案列表
     *
     * @param param 查询条件
     * @return 面试档案列表
     */
    @Override
    public List<InterviewProfileVo> queryList(InterviewProfileListParam param) {
        LambdaQueryWrapper<InterviewProfile> lambdaQuery = this.buildQueryWrapper(param);
        return MapstructUtils.convert(interviewProfileMapper.selectList(lambdaQuery), InterviewProfileVo.class);
    }

    private LambdaQueryWrapper<InterviewProfile> buildQueryWrapper(InterviewProfileListParam param) {
        LambdaQueryWrapper<InterviewProfile> lambdaQuery = Wrappers.<InterviewProfile>lambdaQuery();
        lambdaQuery.orderByDesc(InterviewProfile::getId);
        lambdaQuery.eq(param.getEmployeeId() != null, InterviewProfile::getEmployeeId, param.getEmployeeId());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getJobPosition()), InterviewProfile::getJobPosition, param.getJobPosition());
        lambdaQuery.like(StrUtil.isNotBlank(param.getName()), InterviewProfile::getName, param.getName());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getGender()), InterviewProfile::getGender, param.getGender());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getIdCardNumber()), InterviewProfile::getIdCardNumber, param.getIdCardNumber());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEducation()), InterviewProfile::getEducation, param.getEducation());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getSchool()), InterviewProfile::getSchool, param.getSchool());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getMajor()), InterviewProfile::getMajor, param.getMajor());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPhoneNumber()), InterviewProfile::getPhoneNumber, param.getPhoneNumber());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getBenchmarkEnterprise()), InterviewProfile::getBenchmarkEnterprise, param.getBenchmarkEnterprise());
        lambdaQuery.eq(param.getInterviewDate() != null, InterviewProfile::getInterviewDate, param.getInterviewDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getInterviewSituation()), InterviewProfile::getInterviewSituation, param.getInterviewSituation());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getInterviewResult()), InterviewProfile::getInterviewResult, param.getInterviewResult());
        return lambdaQuery;
    }

    @Override
    public Response importInterviewList(Stream<InterviewProfileImportVo> list) {
        List<InterviewProfileImportVo> errorList = new ArrayList<>();
        List<InterviewProfile> successList = new ArrayList<>();
        list.forEach(item -> {
            if (item.hasError()) {
                errorList.add(item);
                return;
            }
            InterviewProfile training = MapstructUtils.convert(item, InterviewProfile.class);
            successList.add(training);
        });
        if (CollUtil.isNotEmpty(successList)) {
            connectToEmployee(successList);
            interviewProfileMapper.insert(successList);
        }
        StringBuilder message;
        if (CollUtil.isNotEmpty(errorList)) {
            message = new StringBuilder("共" + errorList.size() + "条数据导入失败，错误如下：<br/>");
            errorList.forEach(item -> message.append(item.defaultFailMsg()));
            return Response.buildFailure(ResultCode.FAIL.getCode(), message.toString());
        } else {
            message = new StringBuilder("共" + successList.size() + "条数据导入成功");
            Response response = Response.buildSuccess();
            response.setErrMessage(message.toString());
            return response;
        }
    }

    /**
     * 批量将面试信息关联员工
     *
     * @param list 待关联的面试档案
     */
    private void connectToEmployee(List<InterviewProfile> list) {
        if (CollUtil.isEmpty(list)) {
            return;
        }
        List<InterviewProfile> filter = LambdaUtil.filter(list, item -> item.getEmployeeId() == null);
        if (CollUtil.isEmpty(filter)) {
            return;
        }
        Map<String, InterviewProfile> idCardMap;
        Map<String, InterviewProfile> phoneMap;
        for(List<InterviewProfile> splitedList : CollectionUtils.split(filter, 15)) {
            idCardMap = new HashMap<>(15);
            phoneMap = new HashMap<>(15);
            for (InterviewProfile interviewProfile : splitedList) {
                if (StrUtil.isNotBlank(interviewProfile.getIdCardNumber())) {
                    idCardMap.put(interviewProfile.getIdCardNumber(), interviewProfile);
                } else if (StrUtil.isNotBlank(interviewProfile.getPhoneNumber())) {
                    phoneMap.put(interviewProfile.getPhoneNumber(), interviewProfile);
                }
            }
            if (CollUtil.isNotEmpty(idCardMap.keySet())) {
                List<EmployeeInfo> employeeInfos = employeeInfoMapper.selectList(new LambdaUpdateWrapper<EmployeeInfo>()
                        .in(EmployeeInfo::getIdCardNumber, idCardMap.keySet()));
                if (CollUtil.isNotEmpty(employeeInfos)) {
                    for (EmployeeInfo employeeInfo : employeeInfos) {
                        idCardMap.get(employeeInfo.getIdCardNumber()).setEmployeeId(employeeInfo.getId());
                    }
                }
            }
            if (CollUtil.isNotEmpty(phoneMap.keySet())) {
                List<EmployeeInfo> employeeInfos = employeeInfoMapper.selectList(new LambdaUpdateWrapper<EmployeeInfo>()
                        .in(EmployeeInfo::getPhoneNumber, phoneMap.keySet()));
                if (CollUtil.isNotEmpty(employeeInfos)) {
                    for (EmployeeInfo employeeInfo : employeeInfos) {
                        phoneMap.get(employeeInfo.getPhoneNumber()).setEmployeeId(employeeInfo.getId());
                    }
                }
            }
        }
    }

    /**
     * 查询面试档案
     *
     * @param id 主键
     * @return 面试档案
     */
    @Override
    public InterviewProfileVo queryById(Long id) {
        InterviewProfile interviewProfile = interviewProfileMapper.selectById(id);
        return MapstructUtils.convert(interviewProfile, InterviewProfileVo.class);
    }

    /**
     * 新增面试档案
     *
     * @param param 面试档案
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByParam(InterviewProfileParam param) {
        InterviewProfile interviewProfile = MapstructUtils.convert(param, InterviewProfile.class);
        connectToEmployee(List.of(interviewProfile));
        return this.save(interviewProfile);
    }

    /**
     * 修改面试档案
     *
     * @param param 面试档案
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByParam(InterviewProfileParam param) {
        InterviewProfile interviewProfile = MapstructUtils.convert(param, InterviewProfile.class);
        return this.updateById(interviewProfile);
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(InterviewProfile entity) {
        // 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除面试档案信息
     *
     * @param ids 待删除的主键集合
     * @return 是否删除成功
     */
    @Override
//    @Transactional(rollbackFor = {Exception.class})
    public Boolean delByIds(List<Long> ids) {
        //做一些业务上的校验,判断是否需要校验
        return this.removeByIds(ids);
    }
}
