package com.anplus.hr.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckPermission;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.alibaba.cola.dto.SingleResponse;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.Min;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import top.binfast.common.core.constant.BusinessType;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.core.util.ResponseUtils;
import top.binfast.common.excel.annotion.ExcelExport;
import top.binfast.common.log.annotation.PinSysLog;
import com.anplus.hr.domain.params.StaffingPlanRecordsListParam;
import com.anplus.hr.domain.params.StaffingPlanRecordsParam;
import com.anplus.hr.domain.vo.StaffingPlanRecordsVo;
import com.anplus.hr.service.StaffingPlanRecordsServ;

/**
 * 编制规划记录
 *
 * @author LiuBin
 * @date 2025-12-14
 */
@Validated
@RestController
@RequestMapping("/employee/staffingPlanRecords")
public class StaffingPlanRecordsCtrl {

    @Resource
    private StaffingPlanRecordsServ staffingPlanRecordsServ;

    /**
     * 查询编制规划记录列表
     */
    @SaCheckPermission("employee:staffingPlanRecords:list")
    @GetMapping("/page")
    public PageResponse<StaffingPlanRecordsVo> pageList(StaffingPlanRecordsListParam param) {
        return staffingPlanRecordsServ.queryPageList(param);
    }

    /**
     * 导出编制规划记录列表
     */
    @ExcelExport
    @SaCheckPermission("employee:staffingPlanRecords:export")
    @PinSysLog(value = "编制规划记录", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public List<StaffingPlanRecordsVo> export(StaffingPlanRecordsListParam param) {
        return staffingPlanRecordsServ.queryList(param);
    }

    /**
     * 获取编制规划记录详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("employee:staffingPlanRecords:query")
    @GetMapping("/{id}")
    public SingleResponse<StaffingPlanRecordsVo> getDetail(@PathVariable @Min(1)
                                                    Long id) {
        return SingleResponse.of(staffingPlanRecordsServ.queryById(id));
    }

    /**
     * 新增编制规划记录
     */
    @SaCheckPermission("employee:staffingPlanRecords:add")
    @PinSysLog(value = "编制规划记录", businessType = BusinessType.INSERT)
    @PostMapping()
    public Response add(@Validated(AddGroup.class) @RequestBody StaffingPlanRecordsParam param) {
        return ResponseUtils.ofResult(staffingPlanRecordsServ.insertByParam(param));
    }

    /**
     * 修改编制规划记录
     */
    @SaCheckPermission("employee:staffingPlanRecords:edit")
    @PinSysLog(value = "编制规划记录", businessType = BusinessType.UPDATE)
    @PutMapping()
    public Response edit(@Validated(EditGroup.class) @RequestBody StaffingPlanRecordsParam param) {
        return ResponseUtils.ofResult(staffingPlanRecordsServ.updateByParam(param));
    }

    /**
     * 删除编制规划记录
     *
     * @param ids 主键串
     */
    @SaCheckPermission("employee:staffingPlanRecords:remove")
    @PinSysLog(value = "编制规划记录", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public Response remove(@NotEmpty(message = "主键不能为空")
                           @PathVariable Long[] ids) {
        return ResponseUtils.ofResult(staffingPlanRecordsServ.delByIds(List.of(ids)));
    }
}