package com.anplus.hr.service.impl;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.cola.dto.PageResponse;
import com.anplus.hr.constant.EmployeeChangeLogTypeConstant;
import com.anplus.hr.constant.HrFlowEnum;
import com.anplus.hr.constant.HrFlowTypeConstant;
import com.anplus.hr.constant.HrStatusEnum;
import com.anplus.hr.domain.EmployeeFlow;
import com.anplus.hr.domain.EmployeeInfo;
import com.anplus.hr.domain.params.EmployeeFlowListParam;
import com.anplus.hr.domain.params.EmployeeFlowParam;
import com.anplus.hr.domain.vo.EmployeeFlowVo;
import com.anplus.hr.domain.vo.SysAuditLogVo;
import com.anplus.hr.mapper.EmployeeFlowMapper;
import com.anplus.hr.mapper.EmployeeInfoMapper;
import com.anplus.hr.service.EmployeeAuditLogServ;
import com.anplus.hr.service.EmployeeChangeLogServ;
import com.anplus.hr.service.EmployeeFlowServ;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import top.binfast.app.biz.sysapi.bean.model.auth.SysUser;
import top.binfast.app.biz.sysapi.dao.auth.SysUserMapper;
import top.binfast.common.core.exception.PlatformException;
import top.binfast.common.core.util.MapstructUtils;
import top.binfast.common.core.util.SnGen;
import top.binfast.common.log.model.SysAuditLog;
import top.binfast.common.mybatis.util.QueryUtil;
import top.binfast.daemon.workflow.common.enums.BusinessStatusEnum;
import top.binfast.daemon.workflow.domain.dto.StartProcessDTO;
import top.binfast.daemon.workflow.domain.event.ProcessDeleteEvent;
import top.binfast.daemon.workflow.domain.event.ProcessEvent;
import top.binfast.daemon.workflow.domain.event.ProcessTaskEvent;
import top.binfast.daemon.workflow.service.WorkflowService;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 人事审批对象Service业务层处理
 *
 * @author LiuBin
 * @date 2025-10-31
 */
@Slf4j
@RequiredArgsConstructor
@Service
public class EmployeeFlowServImpl extends ServiceImpl<EmployeeFlowMapper, EmployeeFlow> implements EmployeeFlowServ {

    private final static SnGen snGen = new SnGen();
    private final EmployeeFlowMapper employeeFlowMapper;
    private final WorkflowService workflowService;
    private final EmployeeAuditLogServ employeeAuditLogServ;
    private final EmployeeInfoMapper employeeInfoMapper;
    private final SysUserMapper sysUserMapper;
    private final EmployeeChangeLogServ employeeChangeLogServ;

    /**
     * 分页查询人事审批对象列表
     *
     * @param param 查询条件
     * @return 人事审批对象分页列表
     */
    @Override
    public PageResponse<EmployeeFlowVo> queryPageList(EmployeeFlowListParam param) {
        Page<EmployeeFlow> page = QueryUtil.getPage(param);
        LambdaQueryWrapper<EmployeeFlow> lambdaQuery = this.buildQueryWrapper(param);
        employeeFlowMapper.selectPage(page, lambdaQuery);
        return QueryUtil.getPageResponse(page, MapstructUtils.convert(page.getRecords(), EmployeeFlowVo.class));
    }


    /**
     * 查询符合条件的人事审批对象列表
     *
     * @param param 查询条件
     * @return 人事审批对象列表
     */
    @Override
    public List<EmployeeFlowVo> queryList(EmployeeFlowListParam param) {
        LambdaQueryWrapper<EmployeeFlow> lambdaQuery = this.buildQueryWrapper(param);
        return MapstructUtils.convert(employeeFlowMapper.selectList(lambdaQuery), EmployeeFlowVo.class);
    }

    private LambdaQueryWrapper<EmployeeFlow> buildQueryWrapper(EmployeeFlowListParam param) {
        LambdaQueryWrapper<EmployeeFlow> lambdaQuery = Wrappers.<EmployeeFlow>lambdaQuery();
        lambdaQuery.orderByDesc(EmployeeFlow::getId);
        lambdaQuery.eq(StrUtil.isNotBlank(param.getApplyCode()), EmployeeFlow::getApplyCode, param.getApplyCode());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFlowType()), EmployeeFlow::getFlowType, param.getFlowType());
        lambdaQuery.eq(param.getEmployeeId() != null, EmployeeFlow::getEmployeeId, param.getEmployeeId());
        lambdaQuery.eq(param.getStartDate() != null, EmployeeFlow::getStartDate, param.getStartDate());
        lambdaQuery.eq(param.getEndDate() != null, EmployeeFlow::getEndDate, param.getEndDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getStatus()), EmployeeFlow::getStatus, param.getStatus());
        return lambdaQuery;
    }

    /**
     * 查询人事审批对象
     *
     * @param id 主键
     * @return 人事审批对象
     */
    @Override
    public EmployeeFlowVo queryById(Long id) {
        EmployeeFlow employeeFlow = employeeFlowMapper.selectById(id);
        List<SysAuditLogVo> sysAuditLogVoList = null;
        if (StrUtil.isNotBlank(employeeFlow.getLogIds())) {
            List<SysAuditLog> auditLogs = employeeAuditLogServ.queryAuditLogs(employeeFlow.getLogIds());
            sysAuditLogVoList = MapstructUtils.convert(auditLogs, SysAuditLogVo.class);
        }
        EmployeeFlowVo employeeFlowVo = MapstructUtils.convert(employeeFlow, EmployeeFlowVo.class);
        employeeFlowVo.setAuditLogList(sysAuditLogVoList);
        return employeeFlowVo;
    }

    /**
     * 新增人事审批对象
     *
     * @param param 人事审批对象
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByParam(EmployeeFlowParam param) {
        EmployeeFlow employeeFlow = MapstructUtils.convert(param, EmployeeFlow.class);
        employeeFlow.setId(snGen.nextId());
        return this.save(employeeFlow);
    }

    /**
     * 修改人事审批对象
     *
     * @param param 人事审批对象
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByParam(EmployeeFlowParam param) {
        EmployeeFlow employeeFlow = MapstructUtils.convert(param, EmployeeFlow.class);
        return this.updateById(employeeFlow);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public EmployeeFlowVo submitAndFlowStart(EmployeeFlowParam param) {
//        long day = DateUtil.betweenDay(param.getStartDate(), param.getEndDate(), true);
        if (ObjectUtil.isNull(param.getId())) {
            param.setStatus(BusinessStatusEnum.DRAFT.getStatus());
            param.setApplyCode(System.currentTimeMillis() + StrUtil.EMPTY);
        }
        EmployeeFlow employeeFlow = MapstructUtils.convert(param, EmployeeFlow.class);
        employeeFlow.setId(snGen.nextId());
        boolean flag = baseMapper.insertOrUpdate(employeeFlow);
        if (flag) {
            param.setId(employeeFlow.getId());
            // 后端发起需要忽略权限
            param.getParams().put("ignore", true);

            StartProcessDTO startProcess = new StartProcessDTO();
            startProcess.setBusinessId(employeeFlow.getId().toString());
            startProcess.setFlowCode(StrUtil.isEmpty(param.getFlowCode()) ? "hrEntryFlow" : param.getFlowCode());
            startProcess.setVariables(param.getParams());
            // 后端发起 如果没有登录用户 比如定时任务 可以手动设置一个处理人id
            // startProcess.setHandler("0");

            boolean flag1 = workflowService.startCompleteTask(startProcess);
            if (!flag1) {
                throw new PlatformException("流程发起异常");
            }
        }
        return MapstructUtils.convert(employeeFlow, EmployeeFlowVo.class);
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(EmployeeFlow entity) {
        // 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除人事审批对象信息
     *
     * @param ids 待删除的主键集合
     * @return 是否删除成功
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean delByIds(List<Long> ids) {
        //做一些业务上的校验,判断是否需要校验
        workflowService.deleteInstance(ids);
        return this.removeByIds(ids);
    }

    /**
     * 总体流程监听(例如: 草稿，撤销，退回，作废，终止，已完成等)
     * 正常使用只需#processEvent.flowCode=='leave1'
     * 示例为了方便则使用startsWith匹配了全部示例key
     *
     * @param processEvent 参数
     */
//    @Transactional(rollbackFor = Exception.class)
    @EventListener(condition = "#processEvent.flowCode.startsWith('hr')")
    public void processHandler(ProcessEvent processEvent) {
        log.info("当前任务执行了{}", processEvent.toString());
        EmployeeFlow employeeFlow = baseMapper.selectById(Convert.toLong(processEvent.getBusinessId()));
        employeeFlow.setStatus(processEvent.getStatus());
        // 用于例如审批附件 审批意见等 存储到业务表内 自行根据业务实现存储流程
        Map<String, Object> params = processEvent.getParams();
        if (MapUtil.isNotEmpty(params)) {
            // 历史任务扩展(通常为附件)
            String hisTaskExt = Convert.toStr(params.get("hisTaskExt"));
            // 办理人
            String handler = Convert.toStr(params.get("handler"));
            // 办理意见
            String message = Convert.toStr(params.get("message"));
            if (StrUtil.isNotBlank(handler) && StrUtil.isNotBlank(message)) {
                SysUser sysUser = sysUserMapper.selectById(Convert.toLong(handler));
                if (sysUser != null) {
                    employeeFlow.setHandlerMessage(sysUser.getName() + ":" + message);
                }
            }
        }
        if (processEvent.getSubmit()) {
            employeeFlow.setStatus(BusinessStatusEnum.WAITING.getStatus());
            if (StrUtil.isBlank(employeeFlow.getApplyCode())) {
                String businessCode = MapUtil.getStr(params, "businessCode", StrUtil.EMPTY);
                employeeFlow.setApplyCode(businessCode);
            }
        } else if (processEvent.getStatus().equals(BusinessStatusEnum.FINISH.getStatus())) {
            List<SysAuditLog> auditLogs = new ArrayList<>(6);
            if (StrUtil.isNotBlank(employeeFlow.getLogIds())) {
                auditLogs = employeeAuditLogServ.queryAuditLogs(employeeFlow.getLogIds());
            }
            this.handleProcessFinished(processEvent.getFlowCode(), employeeFlow, auditLogs);
        }
        baseMapper.updateById(employeeFlow);
    }

    /**
     * 执行任务创建监听(也代表上一条任务完成事件)
     * 示例：也可通过  @EventListener(condition = "#processTaskEvent.flowCode=='leave1'")进行判断
     * 在方法中判断流程节点key
     * if ("xxx".equals(processTaskEvent.getNodeCode())) {
     * //执行业务逻辑
     * }
     *
     * @param processTaskEvent 参数
     */
    @EventListener(condition = "#processTaskEvent.flowCode.startsWith('hr')")
    public void processTaskHandler(ProcessTaskEvent processTaskEvent) {
        log.info("当前任务创建了{}", processTaskEvent.toString());
    }

    /**
     * 监听删除流程事件
     * 正常使用只需#processDeleteEvent.flowCode=='leave1'
     * 示例为了方便则使用startsWith匹配了全部示例key
     *
     * @param processDeleteEvent 参数
     */
    @EventListener(condition = "#processDeleteEvent.flowCode.startsWith('hr')")
    public void processDeleteHandler(ProcessDeleteEvent processDeleteEvent) {
        log.info("监听删除流程事件，当前任务执行了{}", processDeleteEvent.toString());
        EmployeeFlow employeeFlow = baseMapper.selectById(Convert.toLong(processDeleteEvent.getBusinessId()));
        if (ObjectUtil.isNull(employeeFlow)) {
            return;
        }
        baseMapper.deleteById(employeeFlow.getId());
    }

    /**
     * 流程结束处理
     *
     * @param flowCode     流程编码
     * @param employeeFlow 员工审批对象
     */
    public void handleProcessFinished(String flowCode, EmployeeFlow employeeFlow, List<SysAuditLog> auditLogs) {
        EmployeeInfo employeeInfo = employeeInfoMapper.selectById(employeeFlow.getEmployeeId());
        switch (flowCode) {
            case HrFlowTypeConstant.ENTRY_CODE:
                // 入职流程结束
                employeeInfo.setEntryApplyStatus(HrFlowEnum.FINISH.getStatus());
                employeeInfo.setStatus(HrStatusEnum.ENTRY.getStatus());
                for (SysAuditLog auditLog : auditLogs) {
                    if ("entryDate".equals(auditLog.getAuditField())) {
                        employeeInfo.setEntryDate(LocalDateTimeUtil.parseDate(auditLog.getAfterVal(), DatePattern.NORM_DATE_PATTERN));
                    }
                }
                employeeChangeLogServ.insertByEmployee(employeeInfo, EmployeeChangeLogTypeConstant.Entry, employeeInfo.getEntryDate());
                break;
            case HrFlowTypeConstant.REGULARIZATION_CODE:
                // 转正流程结束
                employeeInfo.setRegularApplyStatus(HrFlowEnum.FINISH.getStatus());
                employeeInfo.setStatus(HrStatusEnum.REGULARIZATION.getStatus());
                for (SysAuditLog auditLog : auditLogs) {
                    switch (auditLog.getAuditField()) {
                        case "contractStartDate":
                            employeeInfo.setContractStartDate(LocalDateTimeUtil.parseDate(auditLog.getAfterVal(), DatePattern.NORM_DATE_PATTERN));
                            break;
                        case "contractEndDate":
                            employeeInfo.setContractEndDate(LocalDateTimeUtil.parseDate(auditLog.getAfterVal(), DatePattern.NORM_DATE_PATTERN));
                            break;
                        case "contractSigningStatus":
                            employeeInfo.setContractSigningStatus(auditLog.getAfterVal());
                            break;
                        case "contractEntity":
                            employeeInfo.setContractEntity(auditLog.getAfterVal());
                            break;
                        case "socialSecurityEntity":
                            employeeInfo.setSocialSecurityEntity(auditLog.getAfterVal());
                            break;
                        case "hasSocialSecurityPaid":
                            employeeInfo.setHasSocialSecurityPaid(auditLog.getAfterVal());
                            break;
                        case "providentFundEntity":
                            employeeInfo.setProvidentFundEntity(auditLog.getAfterVal());
                            break;
                        case "hasProvidentFundPaid":
                            employeeInfo.setHasProvidentFundPaid(auditLog.getAfterVal());
                            break;
                        case "regularizationDate":
                            employeeInfo.setRegularizationDate(LocalDateTimeUtil.parseDate(auditLog.getAfterVal(), DatePattern.NORM_DATE_PATTERN));
                            break;
                        case "employeeType":
                            employeeInfo.setEmployeeType(auditLog.getAfterVal());
                            break;
                        case "employmentForm":
                            employeeInfo.setEmploymentForm(auditLog.getAfterVal());
                            break;
                        case "contractForm":
                            employeeInfo.setContractForm(auditLog.getAfterVal());
                            break;
                        case "contractTerm":
                            employeeInfo.setContractTerm(auditLog.getAfterVal());
                            break;
                        case "contractExpirationReminder":
                            employeeInfo.setContractExpirationReminder(LocalDateTimeUtil.parseDate(auditLog.getAfterVal(), DatePattern.NORM_DATE_PATTERN));
                            break;
                        default:
                            break;
                    }
                }
                employeeChangeLogServ.insertByEmployee(employeeInfo, EmployeeChangeLogTypeConstant.Regularization, employeeInfo.getRegularizationDate());
                break;
            case HrFlowTypeConstant.RESIGN_CODE:
                // 离职流程结束
                employeeInfo.setResignationApplyStatus(HrFlowEnum.FINISH.getStatus());
                employeeInfo.setStatus(HrStatusEnum.RESIGN.getStatus());
                for (SysAuditLog auditLog : auditLogs) {
                    switch (auditLog.getAuditField()) {
                        case "resignationType":
                            employeeInfo.setResignationType(auditLog.getAfterVal());
                            break;
                        case "resignationReason":
                            employeeInfo.setResignationReason(auditLog.getAfterVal());
                            break;
                        case "resignationDate":
                            employeeInfo.setResignationDate(LocalDateTimeUtil.parseDate(auditLog.getAfterVal(), DatePattern.NORM_DATE_PATTERN));
                            break;
                        case "finalPayDate":
                            employeeInfo.setFinalPayDate(LocalDateTimeUtil.parseDate(auditLog.getAfterVal(), DatePattern.NORM_DATE_PATTERN));
                            break;
                        default:
                            break;
                    }
                }
                employeeChangeLogServ.insertByEmployee(employeeInfo, EmployeeChangeLogTypeConstant.Resign, employeeInfo.getResignationDate());
                break;
            case HrFlowTypeConstant.TRANSFER_CODE:
                // 调配流程结束
                employeeInfo.setTransferApplyStatus(HrFlowEnum.FINISH.getStatus());
                for (SysAuditLog auditLog : auditLogs) {
                    switch (auditLog.getAuditField()) {
                        case "plate":
                            employeeInfo.setPlate(auditLog.getAfterVal());
                            break;
                        case "firstLevelDepartment":
                            employeeInfo.setFirstLevelDepartment(auditLog.getAfterVal());
                            break;
                        case "secondLevelDepartment":
                            employeeInfo.setSecondLevelDepartment(auditLog.getAfterVal());
                            break;
                        case "thirdLevelDepartment":
                            employeeInfo.setThirdLevelDepartment(auditLog.getAfterVal());
                            break;
                        case "deptId":
                            employeeInfo.setDeptId(Convert.toLong(auditLog.getAfterVal()));
                            break;
                        default:
                            break;
                    }
                }
                employeeChangeLogServ.insertByEmployee(employeeInfo, EmployeeChangeLogTypeConstant.Transfer, null);
                break;
            case HrFlowTypeConstant.RENEWAL_CONTRACT:
                // 续签流程结束
                employeeInfo.setRenewalApplyStatus(HrFlowEnum.FINISH.getStatus());
                for (SysAuditLog auditLog : auditLogs) {
                    switch (auditLog.getAuditField()) {
                        case "contractForm":
                            employeeInfo.setContractForm(auditLog.getAfterVal());
                            break;
                        case "contractStartDate":
                            employeeInfo.setContractStartDate(LocalDateTimeUtil.parseDate(auditLog.getAfterVal(), DatePattern.NORM_DATE_PATTERN));
                            break;
                        case "contractEndDate":
                            employeeInfo.setContractEndDate(LocalDateTimeUtil.parseDate(auditLog.getAfterVal(), DatePattern.NORM_DATE_PATTERN));
                            break;
                        case "contractSigningStatus":
                            employeeInfo.setContractSigningStatus(auditLog.getAfterVal());
                            break;
                        case "contractTerm":
                            employeeInfo.setContractTerm(auditLog.getAfterVal());
                            break;
                        case "contractExpirationReminder":
                            employeeInfo.setContractExpirationReminder(LocalDateTimeUtil.parseDate(auditLog.getAfterVal(), DatePattern.NORM_DATE_PATTERN));
                            break;
                        default:
                            break;
                    }
                }
                employeeChangeLogServ.insertByEmployee(employeeInfo, EmployeeChangeLogTypeConstant.RenewalContract, employeeInfo.getContractStartDate());
                break;
            default: {
                // 其他流程结束
                break;
            }
        }
        employeeInfoMapper.updateById(employeeInfo);
    }
}
