package com.anplus.hr.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.anplus.hr.constant.HrStatusEnum;
import com.anplus.hr.domain.EmployeeFamilyMembers;
import com.anplus.hr.domain.EmployeeInfo;
import com.anplus.hr.domain.params.EmployeeInfoListParam;
import com.anplus.hr.domain.params.EmployeeInfoParam;
import com.anplus.hr.domain.vo.EmployeeInfoResignImportVo;
import com.anplus.hr.domain.vo.EmployeeInfoResignVo;
import com.anplus.hr.domain.vo.EmployeeInfoVo;
import com.anplus.hr.mapper.EmployeeFamilyMembersMapper;
import com.anplus.hr.mapper.EmployeeInfoMapper;
import com.anplus.hr.service.EmployeeInfoResignServ;
import com.anplus.hr.service.EmployeeInfoServ;
import com.anplus.hr.service.EmployeeSysDeptServ;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.baomidou.mybatisplus.extension.toolkit.SqlHelper;
import lombok.RequiredArgsConstructor;
import org.dromara.trans.service.impl.TransService;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;
import top.binfast.app.biz.sysapi.bean.model.oss.SysOss;
import top.binfast.app.biz.sysapi.dao.system.SysOssMapper;
import top.binfast.common.core.enums.ResultCode;
import top.binfast.common.core.util.LambdaUtil;
import top.binfast.common.core.util.MapstructUtils;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.excel.core.ExcelContextHolder;
import top.binfast.common.excel.image.ExcelProcessingResult;
import top.binfast.common.excel.image.TempFileExcelImageImporter;
import top.binfast.common.mybatis.util.QueryUtil;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

/**
 * @author 刘斌
 * @date 2025/11/13 20:51
 */
@RequiredArgsConstructor
@Service
public class EmployeeInfoResignServImpl extends ServiceImpl<EmployeeInfoMapper, EmployeeInfo>  implements EmployeeInfoResignServ {

    private final EmployeeInfoMapper employeeInfoMapper;
    private final EmployeeInfoServ employeeInfoServ;
    private final TransService transService;
    private final SysOssMapper sysOssMapper;
    private final EmployeeFamilyMembersMapper employeeFamilyMembersMapper;
    private final EmployeeAsyncService employeeAsyncService;
    private final EmployeeSysDeptServ employeeSysDeptServ;

    /**
     * 分页查询员工信息列表
     *
     */
    @Override
    public PageResponse<EmployeeInfoResignVo> queryPageList(EmployeeInfoListParam param) {
        Page<EmployeeInfo> page = QueryUtil.getPage(param);
        param.setStatus(HrStatusEnum.RESIGN.getStatus());
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = employeeInfoServ.buildQueryWrapper(param);
        employeeInfoMapper.selectPage(page, lambdaQuery);
        return QueryUtil.getPageResponse(page, MapstructUtils.convert(page.getRecords(), EmployeeInfoResignVo.class));
    }

    /**
     * 查询符合条件的员工信息列表
     *
     */
    @Override
    public List<EmployeeInfoResignVo> queryList(EmployeeInfoListParam param) {
        param.setStatus(HrStatusEnum.RESIGN.getStatus());
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = employeeInfoServ.buildQueryWrapper(param);
        List<EmployeeInfoResignVo> employeeInfoResignVos = MapstructUtils.convert(employeeInfoMapper.selectList(lambdaQuery), EmployeeInfoResignVo.class);
        List<Long> ossIdList = LambdaUtil.mapToList(employeeInfoResignVos, EmployeeInfoResignVo::getOssId);
        if (CollUtil.isNotEmpty(ossIdList)) {
            List<SysOss> sysOssList = sysOssMapper.selectByIds(ossIdList);
            Map<Long, String> ossMap = LambdaUtil.toMap(sysOssList, SysOss::getId, SysOss::getFileName);
            for (EmployeeInfoVo employeeInfoVo : employeeInfoResignVos) {
                employeeInfoVo.setPhoto(ossMap.get(employeeInfoVo.getOssId()));
                employeeAsyncService.matchingUrl(employeeInfoVo);
            }
        }
        return employeeInfoResignVos;
    }

    /**
     * 导入离职员工信息列表
     *
     */
    @Override
    public Response importEmployeeResignList(Stream<EmployeeInfoResignImportVo> list, MultipartFile file) {
        List<EmployeeInfoResignImportVo> errorList = new ArrayList<>();
        List<EmployeeInfoResignImportVo> successList = new ArrayList<>();
        List<SysOss> ossList = new ArrayList<>();
        List<EmployeeFamilyMembers> familyMembersList = new ArrayList<>();
        Map<String, EmployeeInfo> ossEmployeeInfoMap = new HashMap<>();
        Map<String, List<EmployeeFamilyMembers>> employeeFamilyMap = new HashMap<>();
        Set<String> idCardNumberSet = new HashSet<>();
        CompletableFuture<ExcelProcessingResult> future = CompletableFuture.supplyAsync(() -> {
            try {
                return TempFileExcelImageImporter.importExcelWithAllImages(file);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
        list.forEach(item -> {
            item.validGroup(EditGroup.class);
            if (item.hasError()) {
                errorList.add(item);
                return;
            }
            EmployeeInfoParam employeeInfoParam = MapstructUtils.convert(item, EmployeeInfoParam.class);
            if (employeeInfoServ.checkEmployeeIdCardNumberUnique(employeeInfoParam) || idCardNumberSet.contains(item.getIdCardNumber())) {
                item.addError("身份证号已存在");
                errorList.add(item);
                return;
            }
            idCardNumberSet.add(item.getIdCardNumber());
            successList.add(item);
        });
        if (CollUtil.isNotEmpty(successList)) {
            Map<String, Long> deptNamesIdMap = employeeSysDeptServ.selectJoinDeptNames();
            ExcelProcessingResult excelProcessingResult = future.join();
            List<EmployeeInfo> insertList = new ArrayList<>(successList.size());
            for (EmployeeInfoResignImportVo importVo : successList) {
                SysOss sysOss = employeeInfoServ.handleImageToUrl(excelProcessingResult, importVo);
                Long leafDeptId = deptNamesIdMap.get(employeeInfoServ.buildDeptNameStr(importVo));
                Long concurrentDeptId = deptNamesIdMap.get(employeeInfoServ.buildPartDeptNameStr(importVo));
//                if (leafDeptId == null) {
//                    importVo.addError("部门不存在");
//                    errorList.add(importVo);
//                    continue;
//                }
                List<EmployeeFamilyMembers> buildFamilyMembersList = employeeInfoServ.buildFamilyMembers(importVo.getFamilyMembersInfo());
                if (!buildFamilyMembersList.isEmpty()) {
                    familyMembersList.addAll(buildFamilyMembersList);
                    employeeFamilyMap.put(importVo.getIdCardNumber(), buildFamilyMembersList);
                }
                EmployeeInfo employeeInfo = MapstructUtils.convert(importVo, EmployeeInfo.class);
                employeeInfo.setStatus(HrStatusEnum.RESIGN.getStatus());
                employeeInfo.setDeptId(leafDeptId);
                employeeInfo.setConcurrentDeptId(concurrentDeptId);
                if (sysOss != null) {
                    ossList.add(sysOss);
                    ossEmployeeInfoMap.put(sysOss.getFileName(), employeeInfo);
                }
                insertList.add(employeeInfo);
            }
            if (CollUtil.isNotEmpty(ossList)) {
                boolean insertFlag = SqlHelper.retBool(sysOssMapper.insert(ossList));
                if (insertFlag) {
                    for (SysOss oss : ossList) {
                        EmployeeInfo employeeInfo = ossEmployeeInfoMap.get(oss.getFileName());
                        if (employeeInfo != null) {
                            employeeInfo.setOssId(oss.getId());
                        }
                    }
                }
            }
            boolean insertEmployeeFlag = SqlHelper.retBool(employeeInfoMapper.insert(insertList));
            if (insertEmployeeFlag && CollUtil.isNotEmpty(familyMembersList)) {
                for (EmployeeInfo employeeInfo : insertList) {
                    List<EmployeeFamilyMembers> existFamilyMembers = employeeFamilyMap.get(employeeInfo.getIdCardNumber());
                    if (existFamilyMembers != null && !existFamilyMembers.isEmpty()) {
                        for (EmployeeFamilyMembers familyMember : existFamilyMembers) {
                            familyMember.setEmployeeId(employeeInfo.getId());
                        }
                    }
                }
                employeeFamilyMembersMapper.insert(familyMembersList);
            }
        }
        StringBuilder message;
        if (CollUtil.isNotEmpty(errorList)) {
            ExcelContextHolder.setErrorExist();
            message = new StringBuilder("共" + errorList.size() + "条数据导入失败，错误如下：<br/>");
            errorList.forEach(item -> message.append(item.defaultFailMsg()));
            return Response.buildFailure(ResultCode.FAIL.getCode(), message.toString());
        } else {
            message = new StringBuilder("共" + successList.size() + "条数据导入成功");
            Response response = Response.buildSuccess();
            response.setErrMessage(message.toString());
            return response;
        }
    }

    /**
     * 获取离职员工信息详情
     */
    @Override
    public EmployeeInfoResignVo infoDetail(Long id) {
        EmployeeInfo employeeInfo = employeeInfoMapper.selectById(id);
        EmployeeInfoResignVo employeeInfoVo = MapstructUtils.convert(employeeInfo, EmployeeInfoResignVo.class);
        if (employeeInfo.getOssId() != null && employeeInfo.getOssId() > 0) {
            SysOss sysOss = sysOssMapper.selectById(employeeInfo.getOssId());
            if (sysOss != null) {
                employeeInfoVo.setPhoto(sysOss.getFileName());
                employeeAsyncService.matchingUrl(employeeInfoVo);
            }
        }
        transService.transOne(employeeInfoVo);
        return employeeInfoVo;
    }
}
