package com.anplus.hr.config;

import java.time.LocalDate;
import java.time.Period;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * 工龄工具类 - 使用字符串分割方法
 * 支持工龄字符串与总月数之间的转换
 */
public class SeniorityUtils {

    // 默认日期格式
    private static final DateTimeFormatter DEFAULT_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    // 常量定义
    private static final String YEAR_KEYWORD = "年";
    private static final String MONTH_KEYWORD = "个月";
    private static final String MONTH_SHORT_KEYWORD = "月";

    /**
     * 将工龄字符串解析为总月数
     * 支持格式: "3年5个月", "3年5月", "5个月", "3年" 等
     */
    public static int parseSeniorityToMonths(String seniorityStr) {
        if (seniorityStr == null || seniorityStr.trim().isEmpty()) {
            return 0;
        }

        String normalized = seniorityStr.trim();
        int years = 0;
        int months = 0;

        try {
            // 检查是否包含年份
            if (normalized.contains(YEAR_KEYWORD)) {
                // 分割年份和月份部分
                String[] yearParts = normalized.split(YEAR_KEYWORD, 2);

                // 解析年份
                years = extractNumber(yearParts[0]);

                // 如果有月份部分
                if (yearParts.length > 1 && !yearParts[1].trim().isEmpty()) {
                    months = extractMonthValue(yearParts[1]);
                }
            } else {
                // 只有月份的情况
                months = extractMonthValue(normalized);
            }

            // 验证数据合理性
            validateSeniority(years, months);

        } catch (Exception e) {
            throw new IllegalArgumentException("无法解析工龄字符串: '" + seniorityStr + "'", e);
        }

        return years * 12 + months;
    }

    /**
     * 从包含月份信息的字符串中提取月份数值
     */
    private static int extractMonthValue(String monthStr) {
        String str = monthStr.trim();

        // 移除月份关键词
        if (str.contains(MONTH_KEYWORD)) {
            str = str.replace(MONTH_KEYWORD, "");
        } else if (str.contains(MONTH_SHORT_KEYWORD)) {
            str = str.replace(MONTH_SHORT_KEYWORD, "");
        }

        // 提取数字
        return extractNumber(str);
    }

    /**
     * 从字符串中提取第一个连续的数字
     */
    private static int extractNumber(String str) {
        if (str == null || str.trim().isEmpty()) {
            return 0;
        }

        String cleaned = str.trim();
        StringBuilder numberBuilder = new StringBuilder();
        boolean foundDigit = false;

        for (char c : cleaned.toCharArray()) {
            if (Character.isDigit(c)) {
                numberBuilder.append(c);
                foundDigit = true;
            } else if (foundDigit) {
                // 遇到非数字字符且已经找到数字，停止提取
                break;
            }
        }

        if (numberBuilder.length() == 0) {
            return 0;
        }

        try {
            return Integer.parseInt(numberBuilder.toString());
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException("无法解析数字: '" + str + "'");
        }
    }

    /**
     * 验证工龄数据的合理性
     */
    private static void validateSeniority(int years, int months) {
        if (years < 0) {
            throw new IllegalArgumentException("工龄年份不能为负数: " + years);
        }
        if (months < 0) {
            throw new IllegalArgumentException("工龄月份不能为负数: " + months);
        }
        if (months >= 12) {
            throw new IllegalArgumentException("工龄月份不能大于等于12: " + months);
        }
    }

    /**
     * 将总月数格式化为工龄字符串
     * 例如: 41个月 -> "3年5个月"
     */
    public static String formatMonthsToSeniority(int totalMonths) {
        return formatMonthsToSeniority(totalMonths, true);
    }

    /**
     * 将总月数格式化为工龄字符串
     * @param totalMonths 总月数
     * @param includeZeroMonths 当月数为0时是否显示 (true: "3年", false: "3年0个月")
     * @return 格式化的工龄字符串
     */
    public static String formatMonthsToSeniority(int totalMonths, boolean includeZeroMonths) {
        if (totalMonths < 0) {
            throw new IllegalArgumentException("总月数不能为负数: " + totalMonths);
        }

        int years = totalMonths / 12;
        int months = totalMonths % 12;

        return buildSeniorityString(years, months, includeZeroMonths);
    }

    /**
     * 构建工龄字符串
     */
    private static String buildSeniorityString(int years, int months, boolean includeZeroMonths) {
        List<String> parts = new ArrayList<>();

        if (years > 0) {
            parts.add(years + YEAR_KEYWORD);
        }

        if (months > 0 || (includeZeroMonths && years == 0)) {
            parts.add(months + MONTH_KEYWORD);
        } else if (months == 0 && !includeZeroMonths && years == 0) {
            // 当总月数为0且不包含0个月时，返回"0个月"
            parts.add("0" + MONTH_KEYWORD);
        }

        if (parts.isEmpty()) {
            return "0" + MONTH_KEYWORD;
        }

        return String.join("", parts);
    }

    /**
     * 将总月数格式化为简化的工龄字符串
     * 例如: 41个月 -> "3年5月" (不使用"个月")
     */
    public static String formatMonthsToShortSeniority(int totalMonths) {
        if (totalMonths < 0) {
            throw new IllegalArgumentException("总月数不能为负数: " + totalMonths);
        }

        int years = totalMonths / 12;
        int months = totalMonths % 12;

        List<String> parts = new ArrayList<>();

        if (years > 0) {
            parts.add(years + YEAR_KEYWORD);
        }

        if (months > 0 || (years == 0 && months == 0)) {
            parts.add(months + MONTH_SHORT_KEYWORD);
        }

        return String.join("", parts);
    }

    /**
     * 解析工龄字符串为年、月数组 [years, months]
     */
    public static int[] parseSeniorityToArray(String seniorityStr) {
        int totalMonths = parseSeniorityToMonths(seniorityStr);
        return new int[]{totalMonths / 12, totalMonths % 12};
    }

    /**
     * 验证工龄字符串格式是否有效
     */
    public static boolean isValidSeniorityFormat(String seniorityStr) {
        if (seniorityStr == null || seniorityStr.trim().isEmpty()) {
            return false;
        }

        try {
            parseSeniorityToMonths(seniorityStr);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 获取工龄的详细描述
     * 例如: 41个月 -> "3年5个月(共41个月)"
     */
    public static String getDetailedDescription(int totalMonths) {
        String seniority = formatMonthsToSeniority(totalMonths);
        return String.format("%s(共%d个月)", seniority, totalMonths);
    }

    /**
     * 比较两个工龄字符串的大小
     * @return 负数: seniority1 < seniority2, 0: 相等, 正数: seniority1 > seniority2
     */
    public static int compareSeniority(String seniority1, String seniority2) {
        int months1 = parseSeniorityToMonths(seniority1);
        int months2 = parseSeniorityToMonths(seniority2);
        return Integer.compare(months1, months2);
    }

    /**
     * 计算两个工龄之间的差异（以月为单位）
     */
    public static int calculateDifference(String seniority1, String seniority2) {
        int months1 = parseSeniorityToMonths(seniority1);
        int months2 = parseSeniorityToMonths(seniority2);
        return Math.abs(months1 - months2);
    }

    /**
     * 添加工龄
     * @param baseSeniority 基础工龄字符串
     * @param yearsToAdd 要添加的年数
     * @param monthsToAdd 要添加的月数
     * @return 新的工龄字符串
     */
    public static String addToSeniority(String baseSeniority, int yearsToAdd, int monthsToAdd) {
        int baseMonths = parseSeniorityToMonths(baseSeniority);
        int totalMonths = baseMonths + yearsToAdd * 12 + monthsToAdd;

        if (totalMonths < 0) {
            throw new IllegalArgumentException("添加工龄后结果不能为负数");
        }

        return formatMonthsToSeniority(totalMonths);
    }

    /**
     * 计算工龄（截止到当前日期）
     * @param employmentDate 入职日期
     * @return 工龄字符串，如 "3年5个月"
     */
    public static String calculateSeniority(LocalDate employmentDate) {
        return calculateSeniority(employmentDate, LocalDate.now());
    }

    /**
     * 计算工龄（指定截止日期）
     * @param employmentDate 入职日期
     * @param endDate 截止日期
     * @return 工龄字符串，如 "3年5个月"
     */
    public static String calculateSeniority(LocalDate employmentDate, LocalDate endDate) {
        Objects.requireNonNull(employmentDate, "入职日期不能为null");
        Objects.requireNonNull(endDate, "截止日期不能为null");

        if (employmentDate.isAfter(endDate)) {
            return "0个月"; // 入职日期在截止日期之后，工龄为0
        }

        Period period = Period.between(employmentDate, endDate);
        int years = period.getYears();
        int months = period.getMonths();

        return formatSeniority(years, months);
    }

    /**
     * 计算工龄并返回总月数（截止到当前日期）
     * @param employmentDate 入职日期
     * @return 总月数
     */
    public static int calculateTotalMonths(LocalDate employmentDate) {
        return calculateTotalMonths(employmentDate, LocalDate.now());
    }

    /**
     * 计算工龄并返回总月数（指定截止日期）
     * @param employmentDate 入职日期
     * @param endDate 截止日期
     * @return 总月数
     */
    public static int calculateTotalMonths(LocalDate employmentDate, LocalDate endDate) {
        Objects.requireNonNull(employmentDate, "入职日期不能为null");
        Objects.requireNonNull(endDate, "截止日期不能为null");

        if (employmentDate.isAfter(endDate)) {
            return 0; // 入职日期在截止日期之后，工龄为0
        }

        Period period = Period.between(employmentDate, endDate);
        int years = period.getYears();
        int months = period.getMonths();

        return years * 12 + months;
    }

    /**
     * 格式化工龄
     * @param years 年数
     * @param months 月数
     * @return 格式化的工龄字符串
     */
    private static String formatSeniority(int years, int months) {
        if (years == 0 && months == 0) {
            return "0个月";
        }

        StringBuilder sb = new StringBuilder();
        if (years > 0) {
            sb.append(years).append("年");
        }
        if (months > 0 || (years == 0 && months == 0)) {
            sb.append(months).append("个月");
        }

        return sb.toString();
    }
}
