package com.anplus.hr.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.alibaba.cola.dto.SingleResponse;
import com.anplus.hr.domain.params.EmployeePartTimeInfoListParam;
import com.anplus.hr.domain.params.EmployeePartTimeInfoParam;
import com.anplus.hr.domain.vo.EmployeePartTimeInfoImportVo;
import com.anplus.hr.domain.vo.EmployeePartTimeInfoVo;
import com.anplus.hr.service.EmployeePartTimeInfoServ;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.Min;
import jakarta.validation.constraints.NotEmpty;
import org.dromara.core.trans.anno.TransMethodResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import top.binfast.common.core.constant.BusinessType;
import top.binfast.common.core.util.ResponseUtils;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.excel.annotion.ExcelExport;
import top.binfast.common.excel.annotion.ExcelStream;
import top.binfast.common.idempotent.annotation.NoRepeatSubmit;
import top.binfast.common.log.annotation.PinSysLog;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

/**
 * 兼职员工信息
 *
 * @author LiuBin
 * @date 2025-12-10
 */
@Validated
@RestController
@RequestMapping("/employee/partTimeInfo")
public class EmployeePartTimeInfoCtrl {

    @Resource
    private EmployeePartTimeInfoServ employeePartTimeInfoServ;

    /**
     * 查询兼职员工信息列表
     */
    @SaCheckPermission("employee:partTimeInfo:list")
    @GetMapping("/page")
    public PageResponse<EmployeePartTimeInfoVo> pageList(EmployeePartTimeInfoListParam param) {
        return employeePartTimeInfoServ.queryPageList(param);
    }

    /**
     * 导出兼职员工信息列表
     */
    @ExcelExport
    @NoRepeatSubmit(interval = 60_000)
    @SaCheckPermission("employee:partTimeInfo:export")
    @PinSysLog(value = "兼职员工信息", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public List<EmployeePartTimeInfoVo> export(EmployeePartTimeInfoListParam param) {
        return employeePartTimeInfoServ.queryList(param);
    }

    /**
     * 导入兼职员工信息列表
     */
//    @NoRepeatSubmit(interval = 60_000)
    @PinSysLog(value = "兼职员工-导入", businessType = BusinessType.IMPORT)
    @SaCheckPermission("employee:partTimeInfo:import")
    @PostMapping(value = "/import")
    public Response importEmployees(@ExcelStream(headRowNumber = 2) Stream<EmployeePartTimeInfoImportVo> stream) {
        return employeePartTimeInfoServ.importPartTimeEmployeeList(stream);
    }

    /**
     * 获取导入模板
     */
    @ExcelExport(template = "兼职模板.xlsx", fill = true)
    @PostMapping("/importTemplate")
    public List<EmployeePartTimeInfoImportVo> importTemplate() {
        return new ArrayList<>();
    }

    /**
     * 获取兼职员工信息详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("employee:partTimeInfo:query")
    @GetMapping("/{id}")
    public SingleResponse<EmployeePartTimeInfoVo> getDetail(@PathVariable @Min(1)
                                                            Long id) {
        return SingleResponse.of(employeePartTimeInfoServ.queryById(id));
    }

    /**
     * 展示员工信息详细信息
     *
     * @param id 主键
     */
    @TransMethodResult
    @SaCheckPermission("employee:partTimeInfo:query")
    @GetMapping("/detail/{id}")
    public SingleResponse<EmployeePartTimeInfoVo> infoDetail(@PathVariable @Min(1)
                                                             Long id) {
        return SingleResponse.of(employeePartTimeInfoServ.infoDetail(id));
    }

    /**
     * 新增兼职员工信息
     */
    @NoRepeatSubmit(interval = 60_000)
    @SaCheckPermission("employee:partTimeInfo:add")
    @PinSysLog(value = "兼职员工信息", businessType = BusinessType.INSERT)
    @PostMapping()
    public Response add(@Validated(AddGroup.class) @RequestBody EmployeePartTimeInfoParam param) {
        return ResponseUtils.ofResult(employeePartTimeInfoServ.insertByParam(param));
    }

    /**
     * 修改兼职员工信息
     */
    @NoRepeatSubmit(interval = 60_000)
    @SaCheckPermission("employee:partTimeInfo:edit")
    @PinSysLog(value = "兼职员工信息", businessType = BusinessType.UPDATE)
    @PutMapping()
    public Response edit(@Validated(EditGroup.class) @RequestBody EmployeePartTimeInfoParam param) {
        return ResponseUtils.ofResult(employeePartTimeInfoServ.updateByParam(param));
    }

    /**
     * 删除兼职员工信息
     *
     * @param ids 主键串
     */
    @NoRepeatSubmit(interval = 60_000)
    @SaCheckPermission("employee:partTimeInfo:remove")
    @PinSysLog(value = "兼职员工信息", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public Response remove(@NotEmpty(message = "主键不能为空")
                           @PathVariable Long[] ids) {
        return ResponseUtils.ofResult(employeePartTimeInfoServ.delByIds(List.of(ids)));
    }
}