package com.anplus.hr.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckPermission;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.alibaba.cola.dto.SingleResponse;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.Min;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import top.binfast.common.core.constant.BusinessType;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.core.util.ResponseUtils;
import top.binfast.common.excel.annotion.ExcelExport;
import top.binfast.common.log.annotation.PinSysLog;
import com.anplus.hr.domain.params.EmployeeChangeLogListParam;
import com.anplus.hr.domain.params.EmployeeChangeLogParam;
import com.anplus.hr.domain.vo.EmployeeChangeLogVo;
import com.anplus.hr.service.EmployeeChangeLogServ;

/**
 * 员工异动记录
 *
 * @author LiuBin
 * @date 2025-11-17
 */
@Validated
@RestController
@RequestMapping("/employee/changeLog")
public class EmployeeChangeLogCtrl {

    @Resource
    private EmployeeChangeLogServ employeeChangeLogServ;

    /**
     * 查询员工异动记录列表
     */
    @SaCheckPermission("employee:changeLog:list")
    @GetMapping("/page")
    public PageResponse<EmployeeChangeLogVo> pageList(EmployeeChangeLogListParam param) {
        return employeeChangeLogServ.queryPageList(param);
    }

    /**
     * 导出员工异动记录列表
     */
    @ExcelExport
    @SaCheckPermission("employee:changeLog:export")
    @PinSysLog(value = "员工异动记录", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public List<EmployeeChangeLogVo> export(EmployeeChangeLogListParam param) {
        return employeeChangeLogServ.queryList(param);
    }

    /**
     * 获取员工异动记录详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("employee:changeLog:query")
    @GetMapping("/{id}")
    public SingleResponse<EmployeeChangeLogVo> getDetail(@PathVariable @Min(1)
                                                    Long id) {
        return SingleResponse.of(employeeChangeLogServ.queryById(id));
    }

    /**
     * 新增员工异动记录
     */
    @SaCheckPermission("employee:changeLog:add")
    @PinSysLog(value = "员工异动记录", businessType = BusinessType.INSERT)
    @PostMapping()
    public Response add(@Validated(AddGroup.class) @RequestBody EmployeeChangeLogParam param) {
        return ResponseUtils.ofResult(employeeChangeLogServ.insertByParam(param));
    }

    /**
     * 修改员工异动记录
     */
    @SaCheckPermission("employee:changeLog:edit")
    @PinSysLog(value = "员工异动记录", businessType = BusinessType.UPDATE)
    @PutMapping()
    public Response edit(@Validated(EditGroup.class) @RequestBody EmployeeChangeLogParam param) {
        return ResponseUtils.ofResult(employeeChangeLogServ.updateByParam(param));
    }

    /**
     * 删除员工异动记录
     *
     * @param ids 主键串
     */
    @SaCheckPermission("employee:changeLog:remove")
    @PinSysLog(value = "员工异动记录", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public Response remove(@NotEmpty(message = "主键不能为空")
                           @PathVariable Long[] ids) {
        return ResponseUtils.ofResult(employeeChangeLogServ.delByIds(List.of(ids)));
    }
}