package com.anplus.hr.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.anplus.hr.config.EmployeeInfoCalcUtils;
import com.anplus.hr.constant.HrEmployeeConstants;
import com.anplus.hr.constant.HrFlowEnum;
import com.anplus.hr.constant.HrFlowTypeConstant;
import com.anplus.hr.constant.HrStatusEnum;
import com.anplus.hr.domain.*;
import com.anplus.hr.domain.params.*;
import com.anplus.hr.domain.vo.*;
import com.anplus.hr.mapper.EmployeeInfoMapper;
import com.anplus.hr.mapper.EmployeeTrainingMapper;
import com.anplus.hr.mapper.InterviewProfileMapper;
import com.anplus.hr.service.EmployeeAuditLogServ;
import com.anplus.hr.service.EmployeeFlowServ;
import com.anplus.hr.service.EmployeeInfoServ;
import com.anplus.hr.service.EmployeeSysDeptServ;
import com.anplus.hr.service.impl.command.EmployeeFamilyMembersCmdExe;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.baomidou.mybatisplus.extension.toolkit.SqlHelper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.dromara.trans.service.impl.TransService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;
import top.binfast.app.biz.sysapi.bean.model.oss.SysOss;
import top.binfast.app.biz.sysapi.bean.vo.sysOss.SysOssVo;
import top.binfast.app.biz.sysapi.dao.system.SysOssMapper;
import top.binfast.app.biz.sysbiz.service.SysOssServ;
import top.binfast.common.core.enums.ResultCode;
import top.binfast.common.core.exception.PlatformException;
import top.binfast.common.core.util.LambdaUtil;
import top.binfast.common.core.util.MapstructUtils;
import top.binfast.common.core.util.StringUtils;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.excel.core.ExcelContextHolder;
import top.binfast.common.excel.image.CellImageData;
import top.binfast.common.excel.image.ExcelProcessingResult;
import top.binfast.common.excel.image.TempFileExcelImageImporter;
import top.binfast.common.log.model.SysAuditLog;
import top.binfast.common.mybatis.util.QueryUtil;
import top.binfast.common.oss.core.OssService;
import top.binfast.common.oss.entity.UploadResult;

import java.time.LocalDate;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

/**
 * 员工信息Service业务层处理
 *
 * @author LiuBin
 * @date 2025-10-28
 */
@Slf4j
@RequiredArgsConstructor
@Service
public class EmployeeInfoServImpl extends ServiceImpl<EmployeeInfoMapper, EmployeeInfo> implements EmployeeInfoServ {

    private final EmployeeInfoMapper employeeInfoMapper;
    private final EmployeeSysDeptServ employeeSysDeptServ;
    private final EmployeeFlowServ employeeFlowServ;
    private final EmployeeAuditLogServ employeeAuditLogServ;
    private final TransService transService;
    private final OssService ossService;
    private final SysOssServ sysOssServ;
    private final SysOssMapper sysOssMapper;
    private final EmployeeFamilyMembersCmdExe familyMembersCmdExe;
    private final EmployeeAsyncService employeeAsyncService;
    private final EmployeeTrainingMapper employeeTrainingMapper;
    private final InterviewProfileMapper interviewProfileMapper;

    /**
     * 分页查询员工信息列表
     *
     * @param param 查询条件
     * @return 员工信息分页列表
     */
    @Override
    public PageResponse<EmployeeInfoVo> queryPageList(EmployeeInfoListParam param) {
        Page<EmployeeInfo> page = QueryUtil.getPage(param);
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = this.buildQueryWrapper(param);
        employeeInfoMapper.selectPage(page, lambdaQuery);
        return QueryUtil.getPageResponse(page, MapstructUtils.convert(page.getRecords(), EmployeeInfoVo.class));
    }


    /**
     * 查询符合条件的员工信息列表
     *
     * @param param 查询条件
     * @return 员工信息列表
     */
    @Override
    public List<EmployeeInfoVo> queryList(EmployeeInfoListParam param) {
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = this.buildQueryWrapper(param);
        List<EmployeeInfoVo> employeeInfoVoList = MapstructUtils.convert(employeeInfoMapper.selectList(lambdaQuery), EmployeeInfoVo.class);
        List<Long> ossIdList = LambdaUtil.mapToList(employeeInfoVoList, EmployeeInfoVo::getOssId);
        if (CollUtil.isNotEmpty(ossIdList)) {
            List<SysOss> sysOssList = sysOssMapper.selectByIds(ossIdList);
            Map<Long, String> ossMap = LambdaUtil.toMap(sysOssList, SysOss::getId, SysOss::getFileName);
            for (EmployeeInfoVo employeeInfoVo : employeeInfoVoList) {
                employeeInfoVo.setPhoto(ossMap.get(employeeInfoVo.getOssId()));
                employeeAsyncService.matchingUrl(employeeInfoVo);
            }
        }
        return employeeInfoVoList;
    }

    @Override
    public LambdaQueryWrapper<EmployeeInfo> buildQueryWrapper(EmployeeInfoListParam param) {
        Map<String, Object> params = param.getParams();
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = Wrappers.<EmployeeInfo>lambdaQuery();
        lambdaQuery.orderByDesc(EmployeeInfo::getId);
        lambdaQuery.ne(EmployeeInfo::getPartTimeFlag, HrEmployeeConstants.YES);
        if (param.getStatus() != null) {
            lambdaQuery.eq(EmployeeInfo::getStatus, param.getStatus());
        } else {
            lambdaQuery.ne(EmployeeInfo::getStatus, HrStatusEnum.RESIGN.getStatus());
        }
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPlate()), EmployeeInfo::getPlate, param.getPlate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFirstLevelDepartment()), EmployeeInfo::getFirstLevelDepartment, param.getFirstLevelDepartment());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getSecondLevelDepartment()), EmployeeInfo::getSecondLevelDepartment, param.getSecondLevelDepartment());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getThirdLevelDepartment()), EmployeeInfo::getThirdLevelDepartment, param.getThirdLevelDepartment());
//        lambdaQuery.eq(param.getDeptId() != null, EmployeeInfo::getDeptId, param.getDeptId());
        if (ObjectUtil.isNotNull(param.getDeptId())) {
            //优先单部门搜索
            lambdaQuery.eq(EmployeeInfo::getDeptId, param.getDeptId());
        } else if (ObjectUtil.isNotNull(param.getBelongDeptId())) {
            //部门树搜索
            lambdaQuery.and(x -> {
                x.in(EmployeeInfo::getDeptId, employeeSysDeptServ.buildQueryWrapper(param.getBelongDeptId()));
            });
        }
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmployeeNo()), EmployeeInfo::getEmployeeNo, param.getEmployeeNo());
        lambdaQuery.eq(param.getJobLevel() != null, EmployeeInfo::getJobLevel, param.getJobLevel());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPositionType()), EmployeeInfo::getPositionType, param.getPositionType());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getSequence()), EmployeeInfo::getSequence, param.getSequence());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPosition()), EmployeeInfo::getPosition, param.getPosition());
        lambdaQuery.like(StrUtil.isNotBlank(param.getName()), EmployeeInfo::getName, param.getName());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getGender()), EmployeeInfo::getGender, param.getGender());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getIdCardNumber()), EmployeeInfo::getIdCardNumber, param.getIdCardNumber());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPhoto()), EmployeeInfo::getPhoto, param.getPhoto());
        lambdaQuery.eq(param.getBirthDate() != null, EmployeeInfo::getBirthDate, param.getBirthDate());
        lambdaQuery.eq(param.getAge() != null, EmployeeInfo::getAge, param.getAge());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getAgeGroup()), EmployeeInfo::getAgeGroup, param.getAgeGroup());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getConcurrentPlate()), EmployeeInfo::getConcurrentPlate, param.getConcurrentPlate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getConcurrentFirstDept()), EmployeeInfo::getConcurrentFirstDept, param.getConcurrentFirstDept());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getConcurrentSecondDept()), EmployeeInfo::getConcurrentSecondDept, param.getConcurrentSecondDept());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getConcurrentThirdDept()), EmployeeInfo::getConcurrentThirdDept, param.getConcurrentThirdDept());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getConcurrentPosition()), EmployeeInfo::getConcurrentPosition, param.getConcurrentPosition());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getNativePlace()), EmployeeInfo::getNativePlace, param.getNativePlace());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEthnicity()), EmployeeInfo::getEthnicity, param.getEthnicity());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getMaritalStatus()), EmployeeInfo::getMaritalStatus, param.getMaritalStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPoliticalStatus()), EmployeeInfo::getPoliticalStatus, param.getPoliticalStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPhoneNumber()), EmployeeInfo::getPhoneNumber, param.getPhoneNumber());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmergencyContact()), EmployeeInfo::getEmergencyContact, param.getEmergencyContact());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmergencyContactPhone()), EmployeeInfo::getEmergencyContactPhone, param.getEmergencyContactPhone());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getHomeAddress()), EmployeeInfo::getHomeAddress, param.getHomeAddress());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getHouseholdRegistrationAddress()), EmployeeInfo::getHouseholdRegistrationAddress, param.getHouseholdRegistrationAddress());
        lambdaQuery.eq(param.getWorkStartDate() != null, EmployeeInfo::getWorkStartDate, param.getWorkStartDate());
        lambdaQuery.between(params.get("entryBeginTime") != null && params.get("entryEndTime") != null,
                EmployeeInfo::getEntryDate, params.get("entryBeginTime"), params.get("entryEndTime"));
        lambdaQuery.eq(StrUtil.isNotBlank(param.getYearsOfService()), EmployeeInfo::getYearsOfService, param.getYearsOfService());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getYearsOfServiceSegment()), EmployeeInfo::getYearsOfServiceSegment, param.getYearsOfServiceSegment());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFulltimeEducation()), EmployeeInfo::getFulltimeEducation, param.getFulltimeEducation());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFulltimeSchool()), EmployeeInfo::getFulltimeSchool, param.getFulltimeSchool());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFulltimeMajor()), EmployeeInfo::getFulltimeMajor, param.getFulltimeMajor());
        lambdaQuery.eq(param.getFulltimeGraduationDate() != null, EmployeeInfo::getFulltimeGraduationDate, param.getFulltimeGraduationDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFulltimeDegree()), EmployeeInfo::getFulltimeDegree, param.getFulltimeDegree());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getNonFulltimeEducation()), EmployeeInfo::getNonFulltimeEducation, param.getNonFulltimeEducation());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getNonFulltimeSchool()), EmployeeInfo::getNonFulltimeSchool, param.getNonFulltimeSchool());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getNonFulltimeMajor()), EmployeeInfo::getNonFulltimeMajor, param.getNonFulltimeMajor());
        lambdaQuery.eq(param.getNonFulltimeGraduationDate() != null, EmployeeInfo::getNonFulltimeGraduationDate, param.getNonFulltimeGraduationDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getNonFulltimeDegree()), EmployeeInfo::getNonFulltimeDegree, param.getNonFulltimeDegree());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEducationCategory()), EmployeeInfo::getEducationCategory, param.getEducationCategory());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmployeeType()), EmployeeInfo::getEmployeeType, param.getEmployeeType());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getProfessionalTitle()), EmployeeInfo::getProfessionalTitle, param.getProfessionalTitle());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getCertificateStatus()), EmployeeInfo::getCertificateStatus, param.getCertificateStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getExternalResume()), EmployeeInfo::getExternalResume, param.getExternalResume());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getInternalResume()), EmployeeInfo::getInternalResume, param.getInternalResume());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmploymentForm()), EmployeeInfo::getEmploymentForm, param.getEmploymentForm());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getContractForm()), EmployeeInfo::getContractForm, param.getContractForm());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getContractTerm()), EmployeeInfo::getContractTerm, param.getContractTerm());
        lambdaQuery.eq(param.getContractStartDate() != null, EmployeeInfo::getContractStartDate, param.getContractStartDate());
        lambdaQuery.eq(param.getContractEndDate() != null, EmployeeInfo::getContractEndDate, param.getContractEndDate());
        lambdaQuery.eq(param.getContractExpirationReminder() != null, EmployeeInfo::getContractExpirationReminder, param.getContractExpirationReminder());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getContractSigningStatus()), EmployeeInfo::getContractSigningStatus, param.getContractSigningStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getContractEntity()), EmployeeInfo::getContractEntity, param.getContractEntity());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getSocialSecurityEntity()), EmployeeInfo::getSocialSecurityEntity, param.getSocialSecurityEntity());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getHasSocialSecurityPaid()), EmployeeInfo::getHasSocialSecurityPaid, param.getHasSocialSecurityPaid());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getProvidentFundEntity()), EmployeeInfo::getProvidentFundEntity, param.getProvidentFundEntity());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getHasProvidentFundPaid()), EmployeeInfo::getHasProvidentFundPaid, param.getHasProvidentFundPaid());
        lambdaQuery.eq(param.getProbationPeriod() != null, EmployeeInfo::getProbationPeriod, param.getProbationPeriod());
        lambdaQuery.eq(param.getRegularizationDate() != null, EmployeeInfo::getRegularizationDate, param.getRegularizationDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getRewardStatus()), EmployeeInfo::getRewardStatus, param.getRewardStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPunishmentStatus()), EmployeeInfo::getPunishmentStatus, param.getPunishmentStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getRemarks()), EmployeeInfo::getRemarks, param.getRemarks());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getOfficePhone()), EmployeeInfo::getOfficePhone, param.getOfficePhone());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getShortLine()), EmployeeInfo::getShortLine, param.getShortLine());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getBankCardNumber()), EmployeeInfo::getBankCardNumber, param.getBankCardNumber());
        lambdaQuery.like(StrUtil.isNotBlank(param.getBankName()), EmployeeInfo::getBankName, param.getBankName());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getHasRelativeInCompany()), EmployeeInfo::getHasRelativeInCompany, param.getHasRelativeInCompany());
        lambdaQuery.like(StrUtil.isNotBlank(param.getRelativeName()), EmployeeInfo::getRelativeName, param.getRelativeName());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getIntroducer()), EmployeeInfo::getIntroducer, param.getIntroducer());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getSalaryLocation()), EmployeeInfo::getSalaryLocation, param.getSalaryLocation());
        lambdaQuery.eq(param.getPerformanceRatio() != null, EmployeeInfo::getPerformanceRatio, param.getPerformanceRatio());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getResignationType()), EmployeeInfo::getResignationType, param.getResignationType());
        lambdaQuery.eq(param.getResignationDate() != null, EmployeeInfo::getResignationDate, param.getResignationDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getResignationReason()), EmployeeInfo::getResignationReason, param.getResignationReason());
        lambdaQuery.eq(param.getFinalPayDate() != null, EmployeeInfo::getFinalPayDate, param.getFinalPayDate());
        lambdaQuery.like(StrUtil.isNotBlank(param.getCostOfDept()), EmployeeInfo::getCostOfDept, param.getCostOfDept());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getCostType()), EmployeeInfo::getCostType, param.getCostType());

//        lambdaQuery.eq(param.getResignationApplyStatus() != null, EmployeeInfo::getResignationApplyStatus, param.getResignationApplyStatus());
//        lambdaQuery.eq(param.getEntryApplyStatus() != null, EmployeeInfo::getEntryApplyStatus, param.getEntryApplyStatus());
//        lambdaQuery.eq(param.getTransferApplyStatus() != null, EmployeeInfo::getTransferApplyStatus, param.getTransferApplyStatus());
        return lambdaQuery;
    }

    /**
     * 导入员工信息列表
     *
     * @param list 员工信息列表
     * @return 是否导入成功
     */
    @Override
    public Response importEmployeeList(Stream<EmployeeInfoImportVo> list, MultipartFile file) {
        List<EmployeeInfoImportVo> errorList = new ArrayList<>();
        List<EmployeeInfoImportVo> successList = new ArrayList<>();
        List<SysOss> ossList = new ArrayList<>();
        List<EmployeeFamilyMembers> familyMembersList = new ArrayList<>();
        Map<String, EmployeeInfo> ossEmployeeInfoMap = new HashMap<>();
        Map<String, List<EmployeeFamilyMembers>> employeeFamilyMap = new HashMap<>();
        Set<String> idCardNumberSet = new HashSet<>();
        CompletableFuture<ExcelProcessingResult> future = CompletableFuture.supplyAsync(() -> {
            try {
                return TempFileExcelImageImporter.importExcelWithAllImages(file);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
        list.forEach(item -> {
            item.validGroup(AddGroup.class);
            if (item.hasError()) {
                errorList.add(item);
                return;
            }
            if (checkEmployeeIdCardNumberUnique(item.getIdCardNumber(), null) ||
                    idCardNumberSet.contains(item.getIdCardNumber())) {
                item.addError("身份证号已存在");
                errorList.add(item);
                return;
            }
            if (item.getContractEndDate() != null) {
                item.setContractExpirationReminder(item.getContractEndDate().minusMonths(1));
            }
            idCardNumberSet.add(item.getIdCardNumber());
            successList.add(item);
        });
        if (CollUtil.isNotEmpty(successList)) {
            Map<String, Long> deptNamesIdMap = employeeSysDeptServ.selectJoinDeptNames();
            ExcelProcessingResult excelProcessingResult = future.join();
            List<EmployeeInfo> insertList = new ArrayList<>(successList.size());
            for (EmployeeInfoImportVo importVo : successList) {
                SysOss sysOss = this.handleImageToUrl(excelProcessingResult, importVo);
                Long leafDeptId = deptNamesIdMap.get(buildDeptNameStr(importVo));
                if (leafDeptId == null) {
                    importVo.addError("部门不存在");
                    errorList.add(importVo);
                    continue;
                }
                Long concurrentDeptId = deptNamesIdMap.get(buildPartDeptNameStr(importVo));
                List<EmployeeFamilyMembers> buildFamilyMembersList = buildFamilyMembers(importVo.getFamilyMembersInfo());
                if (!buildFamilyMembersList.isEmpty()) {
                    familyMembersList.addAll(buildFamilyMembersList);
                    employeeFamilyMap.put(importVo.getIdCardNumber(), buildFamilyMembersList);
                }
                EmployeeInfo employeeInfo = MapstructUtils.convert(importVo, EmployeeInfo.class);
                employeeInfo.setDeptId(leafDeptId);
                employeeInfo.setConcurrentDeptId(concurrentDeptId);
                employeeInfo.setStatus(HrEmployeeConstants.EMPLOYEE_TYPE_REGULAR.equals(employeeInfo.getEmployeeType())
                        ? HrStatusEnum.REGULARIZATION.getStatus() : HrStatusEnum.ENTRY.getStatus());
                EmployeeInfoCalcUtils.calcExpectedRegularDate(employeeInfo);
                if (sysOss != null) {
                    ossList.add(sysOss);
                    ossEmployeeInfoMap.put(sysOss.getFileName(), employeeInfo);
                }
                insertList.add(employeeInfo);
            }
            if (CollUtil.isNotEmpty(ossList)) {
                boolean insertFlag = SqlHelper.retBool(sysOssMapper.insert(ossList));
                if (insertFlag) {
                    for (SysOss oss : ossList) {
                        EmployeeInfo employeeInfo = ossEmployeeInfoMap.get(oss.getFileName());
                        if (employeeInfo != null) {
                            employeeInfo.setOssId(oss.getId());
                        }
                    }
                }
            }
            boolean insertEmployeeFlag = SqlHelper.retBool(employeeInfoMapper.insert(insertList));
            if (insertEmployeeFlag && CollUtil.isNotEmpty(familyMembersList)) {
                for (EmployeeInfo employeeInfo : insertList) {
                    List<EmployeeFamilyMembers> existFamilyMembers = employeeFamilyMap.get(employeeInfo.getIdCardNumber());
                    if (existFamilyMembers != null && !existFamilyMembers.isEmpty()) {
                        for (EmployeeFamilyMembers familyMember : existFamilyMembers) {
                            familyMember.setEmployeeId(employeeInfo.getId());
                        }
                    }
                }
                familyMembersCmdExe.insertBatch(familyMembersList);
            }
//            EmployeeDeptCheckAndSaveParam checkAndSaveParam =
//                    new EmployeeDeptCheckAndSaveParam(deptNameGroups, SecurityUtils.getCurrentUserId(), SecurityUtils.getTenantId());
//            employeeDeptServ.checkAndSaveDept(checkAndSaveParam);
        }
        StringBuilder message;
        if (CollUtil.isNotEmpty(errorList)) {
            ExcelContextHolder.setErrorExist();
            message = new StringBuilder("共" + errorList.size() + "条数据导入失败，错误如下：<br/>");
            errorList.forEach(item -> message.append(item.defaultFailMsg()));
            return Response.buildFailure(ResultCode.FAIL.getCode(), message.toString());
        } else {
            message = new StringBuilder("共" + successList.size() + "条数据导入成功");
            Response response = Response.buildSuccess();
            response.setErrMessage(message.toString());
            return response;
        }
    }

    @Override
    public SysOss handleImageToUrl(ExcelProcessingResult excelProcessingResult, EmployeeInfoImportVo importVo) {
        // 处理图片，保存到OSS，替换成url
        List<CellImageData> imageData = null;
        if (excelProcessingResult.checkWpsOrOffice(importVo.getPhoto())) {
            imageData = excelProcessingResult.getWpsImageData(importVo.getPhoto());
        } else {
            imageData = excelProcessingResult.getOfficeImageData(importVo.getLineNum());
        }
        if (CollUtil.isNotEmpty(imageData)) {
            CellImageData cellImageData = imageData.getFirst();
            UploadResult uploadResult = ossService.upload(cellImageData.getImageName(), cellImageData.getImageData());
//            importVo.setPhoto(uploadResult.getFilename());
//            SysOssExt ext1 = new SysOssExt();
//            ext1.setFileSize((long) cellImageData.getImageData().length);
//            ext1.setContentType(cellImageData.getImageType());
            SysOss sysOss = new SysOss();
            sysOss.setFileName(uploadResult.getFilename());
            sysOss.setOriginalName(uploadResult.getOriginalName());
            sysOss.setFileSuffix(uploadResult.getFileSuffix());
            sysOss.setUrl(uploadResult.getUrl());
            sysOss.setService(ossService.getOssServiceName());
//            sysOss.setExt1(JsonUtils.toJsonString(ext1));
            return sysOss;
        }
        return null;
    }

    @Override
    public SysOssVo upload(MultipartFile file) {
        SysOssVo sysOssVo = sysOssServ.upload(file);
        sysOssMapper.update(null, new LambdaUpdateWrapper<SysOss>()
                .set(SysOss::getTenantId, 1L)
                .eq(SysOss::getId, sysOssVo.getOssId()));
        return sysOssVo;
    }

    @Override
    public Boolean checkEmployeeIdCardNumberUnique(String idCardNumber, Long employeeId) {
        return employeeInfoMapper.exists(new LambdaQueryWrapper<EmployeeInfo>()
                .eq(EmployeeInfo::getIdCardNumber, idCardNumber)
                .ne(ObjectUtil.isNotNull(employeeId), EmployeeInfo::getId, employeeId));
    }

    /**
     * 查询员工信息
     *
     * @param id 主键
     * @return 员工信息
     */
    @Override
    public EmployeeInfoVo queryById(Long id) {
        EmployeeInfo employeeInfo = employeeInfoMapper.selectById(id);
        EmployeeInfoVo employeeInfoVo = MapstructUtils.convert(employeeInfo, EmployeeInfoVo.class);
        familyMembersCmdExe.setFamilyMembersData(employeeInfoVo);
        return employeeInfoVo;
    }

    /**
     * 获取员工信息详细信息
     *
     * @param id 主键
     * @return 员工信息
     */
    @Override
    public EmployeeInfoVo infoDetail(Long id) {
        EmployeeInfo employeeInfo = employeeInfoMapper.selectById(id);
        EmployeeInfoVo employeeInfoVo;
        if (Objects.equals(employeeInfo.getStatus(), HrStatusEnum.RESIGN.getStatus())) {
            employeeInfoVo = MapstructUtils.convert(employeeInfo, EmployeeInfoResignVo.class);
        } else if (HrEmployeeConstants.YES.equals(employeeInfo.getPartTimeFlag())) {
            employeeInfoVo = MapstructUtils.convert(employeeInfo, EmployeePartTimeInfoDetailVo.class);
        } else {
            employeeInfoVo = MapstructUtils.convert(employeeInfo, EmployeeInfoVo.class);
        }
        if (employeeInfo.getOssId() != null && employeeInfo.getOssId() > 0) {
            SysOss sysOss = sysOssMapper.selectById(employeeInfo.getOssId());
            if (sysOss != null) {
                employeeInfoVo.setPhoto(sysOss.getFileName());
                employeeAsyncService.matchingUrl(employeeInfoVo);
            }
        }
        familyMembersCmdExe.setFamilyMembersData(employeeInfoVo);
        transService.transOne(employeeInfoVo);
        return employeeInfoVo;
    }

    /**
     * 获取员工信息详细信息
     * 导出简历用
     *
     * @param id 主键
     * @return 员工信息
     */
    @Override
    public EmployeeInfoResumeVo infoDetailForResume(Long id) {
        EmployeeInfoVo employeeInfoVo = infoDetail(id);
        List<EmployeeTraining> trainings = employeeTrainingMapper.selectListByEmployeeId(id);
        InterviewProfile interviewProfile = interviewProfileMapper.selectByEmployeeId(id);
        List<EmployeeFamilyMembersVo> familyMembersVos = familyMembersCmdExe.selectListByEmployeeId(id);
        EmployeeInfoResumeVo employeeInfoResumeVo = MapstructUtils.convert(employeeInfoVo, EmployeeInfoResumeVo.class);
        employeeInfoResumeVo.setTrainings(MapstructUtils.convert(trainings, EmployeeTrainingVo.class));
        employeeInfoResumeVo.setFamilyMembers(familyMembersVos);
        if (interviewProfile != null) {
            employeeInfoResumeVo.setInterviewDate(interviewProfile.getInterviewDate());
            employeeInfoResumeVo.setInterviewResult(interviewProfile.getInterviewResult());
            employeeInfoResumeVo.setInterviewSituation(interviewProfile.getInterviewSituation());
            employeeInfoResumeVo.setBenchmarkEnterprise(interviewProfile.getBenchmarkEnterprise());
            employeeInfoResumeVo.setJobPosition(interviewProfile.getJobPosition());
        }
        if (StrUtil.isNotBlank(employeeInfoResumeVo.getConcurrentPlate())) {
            employeeInfoResumeVo.setConcurrentFlag("是");
        } else {
            employeeInfoResumeVo.setConcurrentFlag("否");
        }
//        transService.transOne(employeeInfoVo);
        return employeeInfoResumeVo;
    }

    /**
     * 新增员工信息
     *
     * @param param 员工信息
     * @return 是否新增成功
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean insertByParam(EmployeeInfoParam param) {
        EmployeeInfo employeeInfo = MapstructUtils.convert(param, EmployeeInfo.class);
        initDefaultEmployeeInfo(employeeInfo);
//        List<String> deptNameAncestors = employeeSysDeptServ.selectDeptNameAncestorsById(param.getDeptId());
        // 检查或增加部门
//        Set<String> deptNameGroups = new HashSet<>();
//        deptNameGroups.add(employeeInfo.getFirstLevelDepartment() + StrUtil.DASHED + employeeInfo.getSecondLevelDepartment());
//        EmployeeDeptCheckAndSaveParam checkAndSaveParam =
//                new EmployeeDeptCheckAndSaveParam(deptNameGroups, SecurityUtils.getCurrentUserId(), SecurityUtils.getTenantId());
//        employeeDeptServ.checkAndSaveDept(checkAndSaveParam);
//        employeeInfo.setEntryApplyStatus(HrFlowEnum.DRAFT.getStatus());
        employeeInfo.setStatus(HrStatusEnum.DRAFT.getStatus());
        familyMembersCmdExe.updateEmergencyInfo(param.getFamilyMembers(), employeeInfo);
        boolean flag = this.save(employeeInfo);
        if (flag) {
            familyMembersCmdExe.insertByEmployee(param.getFamilyMembers(), employeeInfo);
        }
        return flag;
    }

    private void initDefaultEmployeeInfo(EmployeeInfo employeeInfo) {
        // 计算预计转正时间
        EmployeeInfoCalcUtils.calcExpectedRegularDate(employeeInfo);
        // 更新员工年龄组
        EmployeeInfoCalcUtils.calcAgeGroup(employeeInfo);
        // 计算员工工龄
        EmployeeInfoCalcUtils.calcYearsOfService(employeeInfo);
    }

    /**
     * 修改员工信息
     *
     * @param param 员工信息
     * @return 是否修改成功
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean updateByParam(EmployeeInfoParam param) {
        EmployeeInfo employeeInfo = MapstructUtils.convert(param, EmployeeInfo.class);
        EmployeeInfoCalcUtils.calcYearsOfService(employeeInfo);
        familyMembersCmdExe.updateEmergencyInfo(param.getFamilyMembers(), employeeInfo);
        boolean flag = this.updateById(employeeInfo);
        if (flag) {
            familyMembersCmdExe.updateByEmployee(param.getFamilyMembers(), employeeInfo);
        }
        return flag;
    }

    /**
     * 保存新员工自己填写的基础信息
     *
     * @param param 员工基础信息
     * @return 是否插入成功
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean insertBaseInfoForH5(EmployeeBaseInfoParam param) {
        EmployeeInfo employeeInfo = MapstructUtils.convert(param, EmployeeInfo.class);
        // 更新员工年龄组
        EmployeeInfoCalcUtils.calcAgeGroup(employeeInfo);
//        employeeInfo.setEntryApplyStatus(HrFlowEnum.DRAFT.getStatus());
        employeeInfo.setStatus(HrStatusEnum.DRAFT.getStatus());
        familyMembersCmdExe.updateEmergencyInfo(param.getFamilyMembers(), employeeInfo);
        boolean flag = this.save(employeeInfo);
        if (flag) {
            familyMembersCmdExe.insertByEmployee(param.getFamilyMembers(), employeeInfo);
        }
        return flag;
    }

    /**
     * 员工入职申请
     *
     * @param param 入职申请参数
     * @return 是否申请成功
     */
    @Override
    @DSTransactional(rollbackFor = Exception.class)
    public EmployeeFlowVo applyEntry(EmployeeEntryApplyParam param) {
        if (Convert.toLong(param.getId(), 0L) > 0) {
            checkAndRemoveAuditLogs(param.getId());
        } else {
            EmployeeInfo employeeInfo = employeeInfoMapper.selectById(param.getEmployeeId());
            if (HrFlowEnum.isNotAbleToApply(employeeInfo.getEntryApplyStatus())) {
                throw new PlatformException("当前员工正在申请入职，请勿重复申请");
            }
            employeeInfoMapper.update(null, new LambdaUpdateWrapper<EmployeeInfo>()
                    .set(EmployeeInfo::getEntryApplyStatus, HrFlowEnum.DRAFT.getStatus())
                    .eq(EmployeeInfo::getId, param.getEmployeeId()));
        }
        List<SysAuditLog> list = new ArrayList<>(6);
        employeeAuditLogServ.initAuditLog("入职申请", "entryDate",
                "入职时间", LocalDateTimeUtil.format(param.getEntryDate(), DatePattern.NORM_DATE_PATTERN), list);
        employeeAuditLogServ.saveAuditLogs(list);
        String logIds = LambdaUtil.join(list, (item) -> item.getId() + StrUtil.EMPTY, StrUtil.COMMA);
        EmployeeFlowParam flowParam = new EmployeeFlowParam();
        flowParam.setId(param.getId());
        flowParam.setFlowCode(HrFlowTypeConstant.ENTRY_CODE);
        flowParam.setFlowType(HrFlowTypeConstant.ENTRY);
        flowParam.setEmployeeId(param.getEmployeeId());
        flowParam.setRemark(param.getRemark());
        flowParam.setLogIds(logIds);
        return employeeFlowServ.insertOrUpdateByParam(flowParam);
//        employeeFlowServ.submitAndFlowStart(flowParam);
//        return true;
    }

    /**
     * 员工调岗申请
     *
     * @param param 调岗申请参数
     * @return 是否申请成功
     */
    @Override
    @DSTransactional(rollbackFor = Exception.class)
    public EmployeeFlowVo applyTransfer(EmployeeTransferApplyParam param) {
        if (Convert.toLong(param.getId(), 0L) > 0) {
            checkAndRemoveAuditLogs(param.getId());
        } else {
            EmployeeInfo employeeInfo = employeeInfoMapper.selectById(param.getEmployeeId());
            if (HrFlowEnum.isNotAbleToApply(employeeInfo.getTransferApplyStatus())) {
                throw new PlatformException("当前员工正在申请调配，请勿重复申请");
            }
            employeeInfoMapper.update(null, new LambdaUpdateWrapper<EmployeeInfo>()
                    .set(EmployeeInfo::getTransferApplyStatus, HrFlowEnum.DRAFT.getStatus())
                    .eq(EmployeeInfo::getId, param.getEmployeeId()));
        }
        List<SysAuditLog> list = new ArrayList<>(6);
        employeeAuditLogServ.initAuditLog("调职申请", "plate",
                "板块", param.getPlate(), list);
        employeeAuditLogServ.initAuditLog("调职申请", "firstLevelDepartment",
                "一级部门", param.getFirstLevelDepartment(), list);
        employeeAuditLogServ.initAuditLog("调职申请", "secondLevelDepartment",
                "二级部门", param.getSecondLevelDepartment(), list);
        employeeAuditLogServ.initAuditLog("调职申请", "thirdLevelDepartment",
                "三级部门", param.getThirdLevelDepartment(), list);
        employeeAuditLogServ.initAuditLog("调职申请", "deptId",
                "部门序号", Convert.toStr(param.getDeptId()), list);
        employeeAuditLogServ.initAuditLog("调职申请", "transferDate",
                "调配时间", Convert.toStr(param.getTransferDate()), list);
        employeeAuditLogServ.initAuditLog("调职申请", "performanceRatio",
                "绩效比例", Convert.toStr(param.getPerformanceRatio()), list);
        employeeAuditLogServ.initAuditLog("调职申请", "staffingLocation",
                "编制所在地", Convert.toStr(param.getStaffingLocation()), list);
        employeeAuditLogServ.initAuditLog("调职申请", "costOfDept",
                "成本费用所属", Convert.toStr(param.getCostOfDept()), list);
        employeeAuditLogServ.initAuditLog("调职申请", "costType",
                "成本费用细分", Convert.toStr(param.getCostType()), list);
        employeeAuditLogServ.initAuditLog("调职申请", "internalResume",
                "内部个人履历", Convert.toStr(param.getInternalResume()), list);

        employeeAuditLogServ.saveAuditLogs(list);
        String logIds = LambdaUtil.join(list, (item) -> item.getId() + StrUtil.EMPTY, StrUtil.COMMA);
        EmployeeFlowParam flowParam = new EmployeeFlowParam();
        flowParam.setId(param.getId());
        flowParam.setFlowCode(HrFlowTypeConstant.TRANSFER_CODE);
        flowParam.setFlowType(HrFlowTypeConstant.TRANSFER);
        flowParam.setEmployeeId(param.getEmployeeId());
        flowParam.setRemark(param.getRemark());
        flowParam.setLogIds(logIds);
        return employeeFlowServ.insertOrUpdateByParam(flowParam);
//        employeeFlowServ.submitAndFlowStart(flowParam);
//        return true;
    }

    /**
     * 员工离职申请
     *
     * @param param 入职申请参数
     * @return 是否申请成功
     */
    @Override
    @DSTransactional(rollbackFor = Exception.class)
    public EmployeeFlowVo applyResign(EmployeeResignApplyParam param) {
        if (Convert.toLong(param.getId(), 0L) > 0) {
            checkAndRemoveAuditLogs(param.getId());
        } else {
            EmployeeInfo employeeInfo = employeeInfoMapper.selectById(param.getEmployeeId());
            if (HrFlowEnum.isNotAbleToApply(employeeInfo.getResignationApplyStatus())) {
                throw new PlatformException("当前员工正在申请离职，请勿重复申请");
            }
            employeeInfoMapper.update(null, new LambdaUpdateWrapper<EmployeeInfo>()
                    .set(EmployeeInfo::getResignationApplyStatus, HrFlowEnum.DRAFT.getStatus())
                    .eq(EmployeeInfo::getId, param.getEmployeeId()));
        }
        List<SysAuditLog> list = new ArrayList<>(6);
        employeeAuditLogServ.initAuditLog("离职申请", "resignationType",
                "离职类型", param.getResignationType(), list);
        employeeAuditLogServ.initAuditLog("离职申请", "resignationDate",
                "离职时间", LocalDateTimeUtil.format(param.getResignationDate(), DatePattern.NORM_DATE_PATTERN), list);
        employeeAuditLogServ.initAuditLog("离职申请", "resignationCategory",
                "离职类别", param.getResignationCategory(), list);
        employeeAuditLogServ.initAuditLog("离职申请", "resignationReason",
                "离职原因", param.getResignationReason(), list);
        employeeAuditLogServ.initAuditLog("离职申请", "finalPayDate",
                "最后结薪日", LocalDateTimeUtil.format(param.getFinalPayDate(), DatePattern.NORM_DATE_PATTERN), list);
        employeeAuditLogServ.saveAuditLogs(list);
        String logIds = LambdaUtil.join(list, (item) -> item.getId() + StrUtil.EMPTY, StrUtil.COMMA);
        EmployeeFlowParam flowParam = new EmployeeFlowParam();
        flowParam.setId(param.getId());
        flowParam.setFlowCode(HrFlowTypeConstant.RESIGN_CODE);
        flowParam.setFlowType(HrFlowTypeConstant.RESIGN);
        flowParam.setEmployeeId(param.getEmployeeId());
        flowParam.setRemark(param.getRemark());
        flowParam.setLogIds(logIds);
        return employeeFlowServ.insertOrUpdateByParam(flowParam);
//        employeeFlowServ.submitAndFlowStart(flowParam);
//        return true;
    }

    /**
     * 员工转正申请
     *
     * @param param 转正申请参数
     * @return 是否申请成功
     */
    @Override
    @DSTransactional(rollbackFor = Exception.class)
    public EmployeeFlowVo applyRegular(EmployeeRegularApplyParam param) {
        if (Convert.toLong(param.getId(), 0L) > 0) {
            checkAndRemoveAuditLogs(param.getId());
        } else {
            EmployeeInfo employeeInfo = employeeInfoMapper.selectById(param.getEmployeeId());
            if (HrFlowEnum.isNotAbleToApply(employeeInfo.getRegularApplyStatus())) {
                throw new PlatformException("当前员工正在申请转正，请勿重复申请");
            }
            employeeInfoMapper.update(null, new LambdaUpdateWrapper<EmployeeInfo>()
                    .set(EmployeeInfo::getRegularApplyStatus, HrFlowEnum.DRAFT.getStatus())
                    .eq(EmployeeInfo::getId, param.getEmployeeId()));
        }
        List<SysAuditLog> list = new ArrayList<>(6);
//        String contractStartDateStr = LocalDateTimeUtil.format(param.getContractStartDate(), DatePattern.NORM_DATE_PATTERN);
//        String contractEndDateStr = LocalDateTimeUtil.format(param.getContractEndDate(), DatePattern.NORM_DATE_PATTERN);
//        String contractExpirationReminderStr = LocalDateTimeUtil.format(param.getContractEndDate().minusMonths(1),
//                DatePattern.NORM_DATE_PATTERN);
//        employeeAuditLogServ.initAuditLog("转正申请", "contractStartDate",
//                "劳动合同开始时间", contractStartDateStr, list);
//        employeeAuditLogServ.initAuditLog("转正申请", "contractEndDate",
//                "劳动合同截止时间", contractEndDateStr, list);
//        employeeAuditLogServ.initAuditLog("转正申请", "contractSigningStatus",
//                "劳动合同签订情况", param.getContractSigningStatus(), list);
//        employeeAuditLogServ.initAuditLog("转正申请", "contractEntity",
//                "合同主体", param.getContractEntity(), list);
        employeeAuditLogServ.initAuditLog("转正申请", "regularizationDate",
                "转正时间", LocalDateTimeUtil.format(param.getRegularizationDate(), DatePattern.NORM_DATE_PATTERN), list);
        employeeAuditLogServ.initAuditLog("转正申请", "socialSecurityEntity",
                "社保主体", param.getSocialSecurityEntity(), list);
        employeeAuditLogServ.initAuditLog("转正申请", "hasSocialSecurityPaid",
                "是否缴纳社保", param.getHasSocialSecurityPaid(), list);
        employeeAuditLogServ.initAuditLog("转正申请", "providentFundEntity",
                "公积金主体", param.getProvidentFundEntity(), list);
        employeeAuditLogServ.initAuditLog("转正申请", "hasProvidentFundPaid",
                "是否缴纳公积金", param.getHasProvidentFundPaid(), list);
        employeeAuditLogServ.initAuditLog("转正申请", "hasEmployerInsurancePaid",
                "是否缴纳雇主险", param.getHasEmployerInsurancePaid(), list);

        employeeAuditLogServ.initAuditLog("转正申请", "employeeType",
                "员工类型", HrEmployeeConstants.EMPLOYEE_TYPE_REGULAR, list);
        employeeAuditLogServ.initAuditLog("转正申请", "employmentForm",
                "用工形式", HrEmployeeConstants.EMPLOYMENT_FORM_REGULAR, list);
//        employeeAuditLogServ.initAuditLog("转正申请", "contractForm",
//                "合同形式", HrEmployeeConstants.CONTRACT_FORM_NEW, list);
//        employeeAuditLogServ.initAuditLog("转正申请", "contractTerm",
//                "劳动合同期限", contractStartDateStr + "-" + contractEndDateStr, list);
//        employeeAuditLogServ.initAuditLog("转正申请", "contractExpirationReminder",
//                "合同到期提醒", contractExpirationReminderStr, list);

        employeeAuditLogServ.saveAuditLogs(list);
        String logIds = LambdaUtil.join(list, (item) -> item.getId() + StrUtil.EMPTY, StrUtil.COMMA);
        EmployeeFlowParam flowParam = new EmployeeFlowParam();
        flowParam.setId(param.getId());
        flowParam.setFlowCode(HrFlowTypeConstant.REGULARIZATION_CODE);
        flowParam.setFlowType(HrFlowTypeConstant.REGULARIZATION);
        flowParam.setEmployeeId(param.getEmployeeId());
        flowParam.setRemark(param.getRemark());
        flowParam.setLogIds(logIds);
        return employeeFlowServ.insertOrUpdateByParam(flowParam);
//        employeeFlowServ.submitAndFlowStart(flowParam);
//        return true;
    }

    /**
     * 员工续签合同申请
     *
     * @param param 续签合同申请参数
     * @return 是否申请成功
     */
    @Override
    @DSTransactional(rollbackFor = Exception.class)
    public EmployeeFlowVo applyRenewalContract(EmployeeRenewalContractApplyParam param) {
        if (Convert.toLong(param.getId(), 0L) > 0) {
            checkAndRemoveAuditLogs(param.getId());
        } else {
            EmployeeInfo employeeInfo = employeeInfoMapper.selectById(param.getEmployeeId());
            if (HrFlowEnum.isNotAbleToApply(employeeInfo.getRenewalApplyStatus())) {
                throw new PlatformException("当前员工正在申请合同续约，请勿重复申请");
            }
            employeeInfoMapper.update(null, new LambdaUpdateWrapper<EmployeeInfo>()
                    .set(EmployeeInfo::getRenewalApplyStatus, HrFlowEnum.DRAFT.getStatus())
                    .eq(EmployeeInfo::getId, param.getEmployeeId()));
        }
        List<SysAuditLog> list = new ArrayList<>(6);
        String contractStartDateStr = LocalDateTimeUtil.format(param.getContractStartDate(), DatePattern.NORM_DATE_PATTERN);
        String contractEndDateStr = LocalDateTimeUtil.format(param.getContractEndDate(), DatePattern.NORM_DATE_PATTERN);
        String contractExpirationReminderStr = param.getContractEndDate() == null ? null :
                LocalDateTimeUtil.format(param.getContractEndDate().minusMonths(1), DatePattern.NORM_DATE_PATTERN);
        employeeAuditLogServ.initAuditLog("转正申请", "contractTerm",
                "劳动合同期限", param.getContractTerm(), list);
        employeeAuditLogServ.initAuditLog("转正申请", "contractForm",
                "合同形式", param.getContractForm(), list);
        employeeAuditLogServ.initAuditLog("转正申请", "contractStartDate",
                "劳动合同开始时间", contractStartDateStr, list);
        employeeAuditLogServ.initAuditLog("转正申请", "contractEndDate",
                "劳动合同截止时间", contractEndDateStr, list);
        employeeAuditLogServ.initAuditLog("转正申请", "contractSigningStatus",
                "劳动合同签订情况", param.getContractSigningStatus(), list);

        employeeAuditLogServ.initAuditLog("转正申请", "contractEntity",
                "合同主体", param.getContractEntity(), list);
        employeeAuditLogServ.initAuditLog("转正申请", "contractExpirationReminder",
                "合同到期提醒", contractExpirationReminderStr, list);
        employeeAuditLogServ.initAuditLog("转正申请", "socialSecurityEntity",
                "社保主体", param.getSocialSecurityEntity(), list);
        employeeAuditLogServ.initAuditLog("转正申请", "providentFundEntity",
                "公积金主体", param.getProvidentFundEntity(), list);

        employeeAuditLogServ.saveAuditLogs(list);
        String logIds = LambdaUtil.join(list, (item) -> item.getId() + StrUtil.EMPTY, StrUtil.COMMA);
        EmployeeFlowParam flowParam = new EmployeeFlowParam();
        flowParam.setId(param.getId());
        flowParam.setFlowCode(HrFlowTypeConstant.RENEWAL_CONTRACT_CODE);
        flowParam.setFlowType(HrFlowTypeConstant.RENEWAL_CONTRACT);
        flowParam.setEmployeeId(param.getEmployeeId());
        flowParam.setRemark(param.getRemark());
        flowParam.setLogIds(logIds);
        return employeeFlowServ.insertOrUpdateByParam(flowParam);
//        employeeFlowServ.submitAndFlowStart(flowParam);
//        return true;
    }

    /**
     * 删除原有审核信息
     *
     * @param flowId 主键
     * @return 是否删除成功
     */
    private void checkAndRemoveAuditLogs(Long flowId) {
        EmployeeFlow employeeFlow = employeeFlowServ.getById(flowId);
        if (StrUtil.isNotBlank(employeeFlow.getLogIds())) {
            List<Long> logIds = StringUtils.splitTo(employeeFlow.getLogIds(), Convert::toLong);
            employeeAuditLogServ.removeAuditLogs(logIds);
        }
    }

    /**
     * 校验并批量删除员工信息信息
     *
     * @param ids 待删除的主键集合
     * @return 是否删除成功
     */
    @Override
//    @Transactional(rollbackFor = {Exception.class})
    public Boolean delByIds(List<Long> ids) {
        //做一些业务上的校验,判断是否需要校验
        return this.removeByIds(ids);
    }
}
