package com.anplus.hr.service.impl;

import com.anplus.hr.config.EmployeeInfoCalcUtils;
import com.anplus.hr.constant.HrFlowEnum;
import com.anplus.hr.constant.HrStatusEnum;
import com.anplus.hr.domain.EmployeeInfo;
import com.anplus.hr.mapper.EmployeeInfoMapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import top.binfast.common.sse.dto.SseMessageDto;
import top.binfast.common.sse.utils.SseMessageUtils;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

/**
 * @author 刘斌
 * @date 2025/11/5 17:01
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class EmployeeScheduleService {

    private final EmployeeInfoMapper employeeInfoMapper;

    /**
     * 方式1：使用cron表达式
     * 每天凌晨1点执行
     */
//    @Scheduled(cron = "0 0 1 * * ?")
    public void midnightTask() {
        executeSafely("刷新员工工龄以及年龄", this::processRefreshInfo);
    }


    /**
     * 使用cron表达式
     * 每月15号8点10分执行
     */
//    @Scheduled(cron = "0 10 8 15,16,17,18,19,20 * ?")
    public void monthlyTask() {
        executeSafely("每月检查提醒", this::remindPerMonth);
    }

    /**
     * 每月15到20号号8点10分执行
     */
    private void remindPerMonth() {
        // 实现每月任务逻辑
        boolean existNewRegularReminders = employeeInfoMapper.exists(new LambdaUpdateWrapper<EmployeeInfo>()
                .eq(EmployeeInfo::getStatus, HrStatusEnum.ENTRY.getStatus())
                .isNotNull(EmployeeInfo::getExpectedRegularDate)
                .between(EmployeeInfo::getExpectedRegularDate, LocalDate.now(), LocalDate.now().plusMonths(1))
        );
        if (existNewRegularReminders) {
            SseMessageDto dto = new SseMessageDto();
            dto.setMessage("有新的转正提醒！请检查！");
            dto.setUserIds(List.of(6L));
            SseMessageUtils.publishMessage(dto);
        }
        boolean existContractRenewalReminders = employeeInfoMapper.exists(new LambdaUpdateWrapper<EmployeeInfo>()
                .eq(EmployeeInfo::getStatus, HrStatusEnum.REGULARIZATION.getStatus())
                .isNotNull(EmployeeInfo::getContractEndDate)
                .between(EmployeeInfo::getContractEndDate, LocalDate.now(), LocalDate.now().plusMonths(1))
        );
        if (existContractRenewalReminders) {
            SseMessageDto dto = new SseMessageDto();
            dto.setMessage("有新的合同续约提醒！请检查！");
            dto.setUserIds(List.of(6L));
            SseMessageUtils.publishMessage(dto);
        }
    }

    /**
     * 刷新员工工龄以及工龄组
     */
    private void processRefreshInfo() {
        List<EmployeeInfo> employeeInfos = employeeInfoMapper.selectList(new LambdaUpdateWrapper<EmployeeInfo>()
                .eq(EmployeeInfo::getEntryApplyStatus, HrFlowEnum.FINISH.getStatus())
                .ne(EmployeeInfo::getResignationApplyStatus, HrFlowEnum.FINISH.getStatus()));
        List<EmployeeInfo> updateList = new ArrayList<>();
        for (EmployeeInfo employeeInfo : employeeInfos) {
            boolean isUpdate = EmployeeInfoCalcUtils.calcYearsOfService(employeeInfo);
            isUpdate = isUpdate || EmployeeInfoCalcUtils.calcAgeGroup(employeeInfo);
            if (isUpdate) {
                updateList.add(employeeInfo);
            }
        }
        employeeInfoMapper.updateById(updateList);
    }


    /**
     * 安全执行方法，包含异常处理
     */
    private void executeSafely(String taskName, Runnable task) {
        log.info("开始执行{}", taskName);
        long startTime = System.currentTimeMillis();

        try {
            task.run();
            long cost = System.currentTimeMillis() - startTime;
            log.info("{}执行完成，耗时：{}ms", taskName, cost);
        } catch (Exception e) {
            log.error("{}执行失败", taskName, e);
            // 这里可以添加告警逻辑，如发送邮件、短信等
        }
    }
}
