package com.anplus.hr.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.alibaba.cola.dto.SingleResponse;
import com.anplus.hr.config.StaffingPlanExportHandler;
import com.anplus.hr.domain.params.StaffingPlanDataListParam;
import com.anplus.hr.domain.params.StaffingPlanDataParam;
import com.anplus.hr.domain.vo.StaffingPlanDataImportVo;
import com.anplus.hr.domain.vo.StaffingPlanDataVo;
import com.anplus.hr.domain.vo.StaffingPlanExportVo;
import com.anplus.hr.service.StaffingPlanDataServ;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.Min;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import top.binfast.common.core.constant.BusinessType;
import top.binfast.common.core.util.ResponseUtils;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.excel.annotion.ExcelExport;
import top.binfast.common.excel.annotion.ExcelStream;
import top.binfast.common.log.annotation.PinSysLog;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

/**
 * 编制规划记录明细
 *
 * @author LiuBin
 * @date 2025-12-14
 */
@Validated
@RestController
@RequestMapping("/employee/staffingPlanData")
public class StaffingPlanDataCtrl {

    @Resource
    private StaffingPlanDataServ staffingPlanDataServ;

    /**
     * 查询编制规划记录明细列表
     */
    @SaCheckPermission("employee:staffingPlanData:list")
    @GetMapping("/page")
    public PageResponse<StaffingPlanDataVo> pageList(StaffingPlanDataListParam param) {
        return staffingPlanDataServ.queryPageList(param);
    }

    /**
     * 导出编制规划记录明细列表
     */
    @ExcelExport(template = "编制导出模板.xlsx", fill = true, fillObjectFields = {"dataList"}, writeHandler =
            {StaffingPlanExportHandler.class})
    @SaCheckPermission("employee:staffingPlanData:export")
    @PinSysLog(value = "编制规划记录明细", businessType = BusinessType.EXPORT)
    @PostMapping("/export/{recordId}")
    public List<StaffingPlanExportVo> export(@PathVariable Long recordId) {
        return staffingPlanDataServ.exportList(recordId);
    }

    /**
     * 导入兼职员工信息列表
     */
//    @NoRepeatSubmit(interval = 60_000)
    @PinSysLog(value = "兼职员工-导入", businessType = BusinessType.IMPORT)
    @SaCheckPermission("employee:partTimeInfo:import")
    @PostMapping(value = "/import/{recordId}")
    public Response importEmployees(@ExcelStream(headRowNumber = 2) Stream<StaffingPlanDataImportVo> stream, @PathVariable @Min(1) Long recordId) {
        return staffingPlanDataServ.importStaffingPlanDataList(recordId, stream);
    }

    /**
     * 获取导入模板
     */
    @ExcelExport(template = "编制模板.xlsx", fill = true)
    @PostMapping("/importTemplate")
    public List<StaffingPlanDataVo> importTemplate() {
        return new ArrayList<>();
    }

    /**
     * 获取编制规划记录明细详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("employee:staffingPlanData:query")
    @GetMapping("/{id}")
    public SingleResponse<StaffingPlanDataVo> getDetail(@PathVariable @Min(1)
                                                        Long id) {
        return SingleResponse.of(staffingPlanDataServ.queryById(id));
    }

    /**
     * 新增编制规划记录明细
     */
    @SaCheckPermission("employee:staffingPlanData:add")
    @PinSysLog(value = "编制规划记录明细", businessType = BusinessType.INSERT)
    @PostMapping()
    public Response add(@Validated(AddGroup.class) @RequestBody StaffingPlanDataParam param) {
        return ResponseUtils.ofResult(staffingPlanDataServ.insertByParam(param));
    }

    /**
     * 修改编制规划记录明细
     */
    @SaCheckPermission("employee:staffingPlanData:edit")
    @PinSysLog(value = "编制规划记录明细", businessType = BusinessType.UPDATE)
    @PutMapping()
    public Response edit(@Validated(EditGroup.class) @RequestBody List<StaffingPlanDataParam> param) {
        return ResponseUtils.ofResult(staffingPlanDataServ.updateByParam(param));
    }

    /**
     * 删除编制规划记录明细
     *
     * @param ids 主键串
     */
    @SaCheckPermission("employee:staffingPlanData:remove")
    @PinSysLog(value = "编制规划记录明细", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public Response remove(@NotEmpty(message = "主键不能为空")
                           @PathVariable Long[] ids) {
        return ResponseUtils.ofResult(staffingPlanDataServ.delByIds(List.of(ids)));
    }
}