package com.anplus.hr.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.alibaba.cola.dto.SingleResponse;
import com.anplus.hr.domain.params.InterviewProfileListParam;
import com.anplus.hr.domain.params.InterviewProfileParam;
import com.anplus.hr.domain.vo.InterviewProfileImportVo;
import com.anplus.hr.domain.vo.InterviewProfileVo;
import com.anplus.hr.service.InterviewProfileServ;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.Min;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import top.binfast.common.core.constant.BusinessType;
import top.binfast.common.core.util.ResponseUtils;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.excel.annotion.ExcelExport;
import top.binfast.common.excel.annotion.ExcelStream;
import top.binfast.common.excel.core.ExcelDownHandler;
import top.binfast.common.log.annotation.PinSysLog;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

/**
 * 面试档案
 *
 * @author LiuBin
 * @date 2025-12-08
 */
@Validated
@RestController
@RequestMapping("/employee/interviewProfile")
public class InterviewProfileCtrl {

    @Resource
    private InterviewProfileServ interviewProfileServ;

    /**
     * 查询面试档案列表
     */
    @SaCheckPermission("employee:interviewProfile:list")
    @GetMapping("/page")
    public PageResponse<InterviewProfileVo> pageList(InterviewProfileListParam param) {
        return interviewProfileServ.queryPageList(param);
    }

    /**
     * 导出面试档案列表
     */
    @ExcelExport
    @SaCheckPermission("employee:interviewProfile:export")
    @PinSysLog(value = "面试档案", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public List<InterviewProfileVo> export(InterviewProfileListParam param) {
        return interviewProfileServ.queryList(param);
    }

    /**
     * 导入数据
     */
    @PinSysLog(value = "面试档案", businessType = BusinessType.IMPORT)
    @SaCheckPermission("employee:interviewProfile:import")
    @PostMapping(value = "/import")
    public Response importData(@ExcelStream Stream<InterviewProfileImportVo> stream) {
        return interviewProfileServ.importInterviewList(stream);
    }

    /**
     * 获取导入模板
     */
    @ExcelExport(clazz = InterviewProfileImportVo.class, writeHandler = ExcelDownHandler.class)
    @PostMapping("/importTemplate")
    public List<InterviewProfileImportVo> importTemplate() {
        return new ArrayList<>();
    }

    /**
     * 获取面试档案详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("employee:interviewProfile:query")
    @GetMapping("/{id}")
    public SingleResponse<InterviewProfileVo> getDetail(@PathVariable @Min(1)
                                                        Long id) {
        return SingleResponse.of(interviewProfileServ.queryById(id));
    }

    /**
     * 新增面试档案
     */
    @SaCheckPermission("employee:interviewProfile:add")
    @PinSysLog(value = "面试档案", businessType = BusinessType.INSERT)
    @PostMapping()
    public Response add(@Validated(AddGroup.class) @RequestBody InterviewProfileParam param) {
        return ResponseUtils.ofResult(interviewProfileServ.insertByParam(param));
    }

    /**
     * 修改面试档案
     */
    @SaCheckPermission("employee:interviewProfile:edit")
    @PinSysLog(value = "面试档案", businessType = BusinessType.UPDATE)
    @PutMapping()
    public Response edit(@Validated(EditGroup.class) @RequestBody InterviewProfileParam param) {
        return ResponseUtils.ofResult(interviewProfileServ.updateByParam(param));
    }

    /**
     * 删除面试档案
     *
     * @param ids 主键串
     */
    @SaCheckPermission("employee:interviewProfile:remove")
    @PinSysLog(value = "面试档案", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public Response remove(@NotEmpty(message = "主键不能为空")
                           @PathVariable Long[] ids) {
        return ResponseUtils.ofResult(interviewProfileServ.delByIds(List.of(ids)));
    }
}