package com.anplus.hr.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.anplus.hr.domain.EmployeeInfo;
import com.anplus.hr.domain.EmployeeTraining;
import com.anplus.hr.domain.params.EmployeeTrainingListParam;
import com.anplus.hr.domain.params.EmployeeTrainingParam;
import com.anplus.hr.domain.vo.EmployeeTrainingImportVo;
import com.anplus.hr.domain.vo.EmployeeTrainingVo;
import com.anplus.hr.mapper.EmployeeInfoMapper;
import com.anplus.hr.mapper.EmployeeTrainingMapper;
import com.anplus.hr.service.EmployeeTrainingServ;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import top.binfast.common.core.enums.ResultCode;
import top.binfast.common.core.util.LambdaUtil;
import top.binfast.common.core.util.MapstructUtils;
import top.binfast.common.mybatis.util.QueryUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

/**
 * 员工培训档案Service业务层处理
 *
 * @author LiuBin
 * @date 2025-12-07
 */
@RequiredArgsConstructor
@Service
public class EmployeeTrainingServImpl extends ServiceImpl<EmployeeTrainingMapper, EmployeeTraining> implements EmployeeTrainingServ {

    private final EmployeeTrainingMapper employeeTrainingMapper;
    private final EmployeeInfoMapper employeeInfoMapper;

    /**
     * 分页查询员工培训档案列表
     *
     * @param param 查询条件
     * @return 员工培训档案分页列表
     */
    @Override
    public PageResponse<EmployeeTrainingVo> queryPageList(EmployeeTrainingListParam param) {
        Page<EmployeeTraining> page = QueryUtil.getPage(param);
        LambdaQueryWrapper<EmployeeTraining> lambdaQuery = this.buildQueryWrapper(param);
        employeeTrainingMapper.selectPage(page, lambdaQuery);
        return QueryUtil.getPageResponse(page, MapstructUtils.convert(page.getRecords(), EmployeeTrainingVo.class));
    }


    /**
     * 查询符合条件的员工培训档案列表
     *
     * @param param 查询条件
     * @return 员工培训档案列表
     */
    @Override
    public List<EmployeeTrainingVo> queryList(EmployeeTrainingListParam param) {
        LambdaQueryWrapper<EmployeeTraining> lambdaQuery = this.buildQueryWrapper(param);
        return MapstructUtils.convert(employeeTrainingMapper.selectList(lambdaQuery), EmployeeTrainingVo.class);
    }

    private LambdaQueryWrapper<EmployeeTraining> buildQueryWrapper(EmployeeTrainingListParam param) {
        LambdaQueryWrapper<EmployeeTraining> lambdaQuery = Wrappers.<EmployeeTraining>lambdaQuery();
        lambdaQuery.orderByDesc(EmployeeTraining::getId);
//        lambdaQuery.eq(param.getEmployeeId() != null, EmployeeTraining::getEmployeeId, param.getEmployeeId());
        lambdaQuery.like(StrUtil.isNotBlank(param.getName()), EmployeeTraining::getName, param.getName());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getTrainingCourse()), EmployeeTraining::getTrainingCourse, param.getTrainingCourse());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getTrainingInstitution()), EmployeeTraining::getTrainingInstitution, param.getTrainingInstitution());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getTrainer()), EmployeeTraining::getTrainer, param.getTrainer());
        lambdaQuery.eq(param.getTrainingTime() != null, EmployeeTraining::getTrainingTime, param.getTrainingTime());
        lambdaQuery.eq(param.getServiceEndDate() != null, EmployeeTraining::getServiceEndDate, param.getServiceEndDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getServiceYears()), EmployeeTraining::getServiceYears, param.getServiceYears());
        lambdaQuery.eq(param.getTrainingAmount() != null, EmployeeTraining::getTrainingAmount, param.getTrainingAmount());
        return lambdaQuery;
    }

    @Override
    public Response importTrainingList(Stream<EmployeeTrainingImportVo> list) {
        List<EmployeeTrainingImportVo> errorList = new ArrayList<>();
        List<EmployeeTraining> successList = new ArrayList<>();
        list.forEach(item -> {
            if (item.hasError()) {
                errorList.add(item);
                return;
            }
            EmployeeTraining training = MapstructUtils.convert(item, EmployeeTraining.class);
            successList.add(training);
        });
        if (CollUtil.isNotEmpty(successList)) {
            connectToEmployee(successList);
            employeeTrainingMapper.insert(successList);
        }
        StringBuilder message;
        if (CollUtil.isNotEmpty(errorList)) {
            message = new StringBuilder("共" + errorList.size() + "条数据导入失败，错误如下：<br/>");
            errorList.forEach(item -> message.append(item.defaultFailMsg()));
            return Response.buildFailure(ResultCode.FAIL.getCode(), message.toString());
        } else {
            message = new StringBuilder("共" + successList.size() + "条数据导入成功");
            Response response = Response.buildSuccess();
            response.setErrMessage(message.toString());
            return response;
        }
    }

    /**
     * 连接员工培训档案和员工信息
     *
     * @param list 员工培训档案列表
     */
    private void connectToEmployee(List<EmployeeTraining> list) {
        if (CollUtil.isEmpty(list)) {
            return;
        }
        List<EmployeeTraining> filter = LambdaUtil.filter(list, item -> item.getEmployeeId() == null);
        if (CollUtil.isEmpty(filter)) {
            return;
        }
        Map<String, EmployeeTraining> idCardMap;
        for (List<EmployeeTraining> splitedList : CollectionUtils.split(filter, 15)) {
            idCardMap = LambdaUtil.filterToMap(splitedList, item -> StrUtil.isNotBlank(item.getIdCardNumber()),
                    EmployeeTraining::getIdCardNumber, item -> item);
            if (CollUtil.isNotEmpty(idCardMap.keySet())) {
                List<EmployeeInfo> employeeInfos = employeeInfoMapper.selectList(new LambdaUpdateWrapper<EmployeeInfo>()
                        .in(EmployeeInfo::getIdCardNumber, idCardMap.keySet()));
                if (CollUtil.isNotEmpty(employeeInfos)) {
                    for (EmployeeInfo employeeInfo : employeeInfos) {
                        idCardMap.get(employeeInfo.getIdCardNumber()).setEmployeeId(employeeInfo.getId());
                    }
                }
            }
        }
    }


    /**
     * 查询员工培训档案
     *
     * @param id 主键
     * @return 员工培训档案
     */
    @Override
    public EmployeeTrainingVo queryById(Long id) {
        EmployeeTraining employeeTraining = employeeTrainingMapper.selectById(id);
        return MapstructUtils.convert(employeeTraining, EmployeeTrainingVo.class);
    }

    /**
     * 新增员工培训档案
     *
     * @param param 员工培训档案
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByParam(EmployeeTrainingParam param) {
        EmployeeTraining employeeTraining = MapstructUtils.convert(param, EmployeeTraining.class);
        connectToEmployee(List.of(employeeTraining));
        return this.save(employeeTraining);
    }

    /**
     * 修改员工培训档案
     *
     * @param param 员工培训档案
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByParam(EmployeeTrainingParam param) {
        EmployeeTraining employeeTraining = MapstructUtils.convert(param, EmployeeTraining.class);
        return this.updateById(employeeTraining);
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(EmployeeTraining entity) {
        // 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除员工培训档案信息
     *
     * @param ids 待删除的主键集合
     * @return 是否删除成功
     */
    @Override
//    @Transactional(rollbackFor = {Exception.class})
    public Boolean delByIds(List<Long> ids) {
        //做一些业务上的校验,判断是否需要校验
        return this.removeByIds(ids);
    }
}
